C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - NOVEMBER 1, 1984
C
C   PURPOSE             - DIFFERENTIAL EQUATION SOLVER - VARIABLE ORDER
C                           ADAMS PREDICTOR CORRECTOR METHOD OR
C                           GEARS METHOD
C
C   USAGE               - CALL DGEAR (N,FCN,FCNJ,X,H,Y,XEND,TOL,METH,
C                           MITER,INDEX,IWK,WK,IER)
C
C   ARGUMENTS    N      - INPUT NUMBER OF FIRST-ORDER DIFFERENTIAL
C                           EQUATIONS.
C                FCN    - NAME OF SUBROUTINE FOR EVALUATING FUNCTIONS.
C                           (INPUT)
C                           THE SUBROUTINE ITSELF MUST ALSO BE PROVIDED
C                             BY THE USER AND IT SHOULD BE OF THE
C                             FOLLOWING FORM
C                               SUBROUTINE FCN (N,X,Y,YPRIME)
C                               REAL X,Y(N),YPRIME(N)
C                                    .
C                                    .
C                                    .
C                           FCN SHOULD EVALUATE YPRIME(1),...,YPRIME(N)
C                             GIVEN N,X, AND Y(1),...,Y(N). YPRIME(I)
C                             IS THE FIRST DERIVATIVE OF Y(I) WITH
C                             RESPECT TO X.
C                           FCN MUST APPEAR IN AN EXTERNAL STATEMENT IN
C                             THE CALLING PROGRAM AND N,X,Y(1),...,Y(N)
C                             MUST NOT BE ALTERED BY FCN.
C                FCNJ   - NAME OF THE SUBROUTINE FOR COMPUTING THE
C                           JACOBIAN MATRIX OF PARTIAL DERIVATIVES.
C                           (INPUT)
C                           THE SUBROUTINE ITSELF MUST ALSO BE PROVIDED
C                             BY THE USER.
C                           IF MITER=1 IT SHOULD BE OF THE FOLLOWING
C                             FORM
C                               SUBROUTINE FCNJ (N,X,Y,PD)
C                               REAL X,Y(N),PD(N,N)
C                                    .
C                                    .
C                           FCNJ MUST EVALUATE PD(I,J), THE PARTIAL
C                             DERIVATIVE OF YPRIME(I) WITH RESPECT TO
C                             Y(J), FOR I=1,N AND J=1,N.
C                           IF MITER= -1 IT SHOULD BE OF THE FOLLOWING
C                             FORM
C                               SUBROUTINE FCNJ (N,X,Y,PD)
C                               REAL X,Y(N),PD(1)
C                                    .
C                                    .
C                           FCNJ MUST EVALUATE PD IN BAND STORAGE MODE.
C                             THAT IS, PD(N*(J-I+NLC)+I) IS THE PARTIAL
C                             DERIVATIVE OF YPRIME(I) WITH RESPECT TO
C                             Y(J).  NLC IS THE NUMBER OF LOWER
C                             CODIAGONALS FOR THE BAND MATRIX.
C                           FCNJ MUST APPEAR IN AN EXTERNAL STATEMENT IN
C                             THE CALLING PROGRAM AND N,X,Y(1),...,Y(N)
C                             MUST NOT BE ALTERED BY FCNJ.
C                           FCNJ IS USED ONLY IF MITER IS EQUAL TO
C                             1 OR -1. OTHERWISE A DUMMY ROUTINE CAN
C                             BE SUBSTITUTED. SEE REMARK 1.
C                X      - INDEPENDENT VARIABLE. (INPUT AND OUTPUT)
C                           ON INPUT, X SUPPLIES THE INITIAL VALUE
C                             AND IS USED ONLY ON THE FIRST CALL.
C                           ON OUTPUT, X IS REPLACED WITH THE CURRENT
C                             VALUE OF THE INDEPENDENT VARIABLE AT WHICH
C                             INTEGRATION HAS BEEN COMPLETED.
C                H      - INPUT/OUTPUT.
C                           ON INPUT, H CONTAINS THE NEXT STEP SIZE IN
C                             X. H IS USED ONLY ON THE FIRST CALL.
C                           ON OUTPUT, H CONTAINS THE STEP SIZE USED
C                             LAST, WHETHER SUCCESSFULLY OR NOT.
C                Y      - DEPENDENT VARIABLES, VECTOR OF LENGTH N.
C                           (INPUT AND OUTPUT)
C                           ON INPUT, Y(1),...,Y(N) SUPPLY INITIAL
C                             VALUES.
C                           ON OUTPUT, Y(1),...,Y(N) ARE REPLACED WITH
C                             A COMPUTED VALUE AT XEND.
C                XEND   - INPUT VALUE OF X AT WHICH SOLUTION IS DESIRED
C                           NEXT. INTEGRATION WILL NORMALLY GO
C                           BEYOND XEND AND THE ROUTINE WILL INTERPOLATE
C                           TO X = XEND.
C                         NOTE THAT (X-XEND)*H MUST BE LESS THAN
C                           ZERO (X AND H AS SPECIFIED ON INPUT).
C                TOL    - INPUT RELATIVE ERROR BOUND. TOL MUST BE
C                           GREATER THAN ZERO. TOL IS USED ONLY ON THE
C                           FIRST CALL UNLESS INDEX IS EQUAL TO -1.
C                           TOL SHOULD BE AT LEAST AN ORDER OF
C                           MAGNITUDE LARGER THAN THE UNIT ROUNDOFF
C                           BUT GENERALLY NOT LARGER THAN .001.
C                           SINGLE STEP ERROR ESTIMATES DIVIDED BY
C                           YMAX(I) WILL BE KEPT LESS THAN TOL IN
C                           ROOT-MEAN-SQUARE NORM (EUCLIDEAN NORM
C                           DIVIDED BY SQRT(N)). THE VECTOR YMAX OF
C                           WEIGHTS IS COMPUTED INTERNALLY AND STORED
C                           IN WORK VECTOR WK. INITIALLY YMAX(I) IS
C                           THE ABSOLUTE VALUE OF Y(I), WITH A DEFAULT
C                           VALUE OF ONE IF Y(I) IS EQUAL TO ZERO.
C                           THEREAFTER, YMAX(I) IS THE LARGEST VALUE
C                           OF THE ABSOLUTE VALUE OF Y(I) SEEN SO FAR,
C                           OR THE INITIAL VALUE OF YMAX(I) IF THAT IS
C                           LARGER.
C                METH   - INPUT BASIC METHOD INDICATOR.
C                           USED ONLY ON THE FIRST CALL UNLESS INDEX IS
C                           EQUAL TO -1.
C                         METH = 1, IMPLIES THAT THE ADAMS METHOD IS
C                           TO BE USED.
C                         METH = 2, IMPLIES THAT THE STIFF METHODS OF
C                           GEAR, OR THE BACKWARD DIFFERENTIATION
C                           FORMULAE ARE TO BE USED.
C                MITER  - INPUT ITERATION METHOD INDICATOR.
C                           MITER = 0, IMPLIES THAT FUNCTIONAL
C                             ITERATION IS USED. NO PARTIAL
C                             DERIVATIVES ARE NEEDED. A DUMMY FCNJ
C                             CAN BE USED.
C                           MITER = 1, IMPLIES THAT THE CHORD METHOD
C                             IS USED WITH AN ANALYTIC JACOBIAN. FOR
C                             THIS METHOD, THE USER SUPPLIES
C                             SUBROUTINE FCNJ.
C                           MITER = 2, IMPLIES THAT THE CHORD METHOD
C                             IS USED WITH THE JACOBIAN CALCULATED
C                             INTERNALLY BY FINITE DIFFERENCES.
C                             A DUMMY FCNJ CAN BE USED.
C                           MITER = 3, IMPLIES THAT THE CHORD METHOD
C                             IS USED WITH THE JACOBIAN REPLACED BY
C                             A DIAGONAL APPROXIMATION BASED ON A
C                             DIRECTIONAL DERIVATIVE.
C                             A DUMMY FCNJ CAN BE USED.
C                           MITER = -1 OR -2, IMPLIES USE THE SAME
C                             METHOD AS FOR MITER= 1 OR 2, RESPECTIVELY,
C                             BUT USING A BANDED JACOBIAN MATRIX.  IN
C                             THESE TWO CASES BANDWIDTH INFORMATION
C                             MUST BE PASSED TO DGEAR THROUGH THE
C                             COMMON BLOCK
C                                COMMON /DBAND/ NLC,NUC
C                             WHERE NLC=NUMBER OF LOWER CODIAGONALS
C                                   NUC=NUMBER OF UPPER CODIAGONALS
C                INDEX  - INPUT AND OUTPUT PARAMETER USED TO INDICATE
C                           THE TYPE OF CALL TO THE SUBROUTINE.  ON
C                           OUTPUT INDEX IS RESET TO 0 IF INTEGRATION
C                           WAS SUCCESSFUL.  OTHERWISE, THE VALUE OF
C                           INDEX IS UNCHANGED.
C                         ON INPUT, INDEX = 1, IMPLIES THAT THIS IS THE
C                           FIRST CALL FOR THIS PROBLEM.
C                         ON INPUT, INDEX = 0, IMPLIES THAT THIS IS NOT
C                           THE FIRST CALL FOR THIS PROBLEM.
C                         ON INPUT, INDEX = -1, IMPLIES THAT THIS IS NOT
C                           THE FIRST CALL FOR THIS PROBLEM, AND THE
C                           USER HAS RESET TOL.
C                         ON INPUT, INDEX = 2, IMPLIES THAT THIS IS NOT
C                           THE FIRST CALL FOR THIS PROBLEM. INTEGRATION
C                           IS TO CONTINUE AND XEND IS TO BE HIT EXACTLY
C                           (NO INTERPOLATION IS DONE). THIS VALUE OF
C                           INDEX ASSUMES THAT XEND IS BEYOND THE
C                           CURRENT VALUE OF X.
C                         ON INPUT, INDEX = 3, IMPLIES THAT THIS IS NOT
C                           THE FIRST CALL FOR THIS PROBLEM. INTEGRATION
C                           IS TO CONTINUE AND CONTROL IS TO BE RETURNED
C                           TO THE CALLING PROGRAM AFTER ONE STEP. XEND
C                           IS IGNORED.
C                IWK    - INTEGER WORK VECTOR OF LENGTH N. USED ONLY IF
C                           MITER = 1 OR 2
C                WK     - REAL WORK VECTOR OF LENGTH 4*N+NMETH+NMITER.
C                           THE VALUE OF NMETH DEPENDS ON THE VALUE OF
C                             METH.
C                             IF METH IS EQUAL TO 1,
C                               NMETH IS EQUAL TO N*13.
C                             IF METH IS EQUAL TO 2,
C                               NMETH IS EQUAL TO N*6.
C                           THE VALUE OF NMITER DEPENDS ON THE VALUE OF
C                             MITER.
C                             IF MITER IS EQUAL TO 1 OR 2,
C                               NMITER IS EQUAL TO N*(N+1)
C                             IF MITER IS EQUAL TO -1 OR -2,
C                               NMITER IS EQUAL TO (2*NLC+NUC+3)*N
C                                WHERE NLC=NUMBER OF LOWER CODIAGONALS
C                                      NUC=NUMBER OF UPPER CODIAGONALS
C                             IF MITER IS EQUAL TO 3,
C                               NMITER IS EQUAL TO N.
C                             IF MITER IS EQUAL TO 0,
C                               NMITER IS EQUAL TO 1.
C                           WK MUST REMAIN UNCHANGED BETWEEN SUCCESSIVE
C                           CALLS DURING INTEGRATION.
C                IER    - ERROR PARAMETER. (OUTPUT)
C                         WARNING ERROR
C                           IER = 33, IMPLIES THAT X+H WILL EQUAL X ON
C                             THE NEXT STEP. THIS CONDITION DOES NOT
C                             FORCE THE ROUTINE TO HALT. HOWEVER, IT
C                             DOES INDICATE ONE OF TWO CONDITIONS.
C                             THE USER MIGHT BE REQUIRING TOO MUCH
C                             ACCURACY VIA THE INPUT PARAMETER TOL.
C                             IN THIS CASE THE USER SHOULD CONSIDER
C                             INCREASING THE VALUE OF TOL. THE OTHER
C                             CONDITION WHICH MIGHT GIVE RISE TO THIS
C                             ERROR MESSAGE IS THAT THE SYSTEM OF
C                             DIFFERENTIAL EQUATIONS BEING SOLVED
C                             IS STIFF (EITHER IN GENERAL OR OVER
C                             THE SUBINTERVAL OF THE PROBLEM BEING
C                             SOLVED AT THE TIME OF THE ERROR). IN
C                             THIS CASE THE USER SHOULD CONSIDER
C                             USING A NONZERO VALUE FOR THE INPUT
C                             PARAMETER MITER.
C                         WARNING WITH FIX ERROR
C                           IER = 66, IMPLIES THAT THE ERROR TEST
C                             FAILED. H WAS REDUCED BY .1 ONE OR MORE
C                             TIMES AND THE STEP WAS TRIED AGAIN
C                             SUCCESSFULLY.
C                           IER = 67, IMPLIES THAT CORRECTOR
C                             CONVERGENCE COULD NOT BE ACHIEVED.
C                             H WAS REDUCED BY .1 ONE OR MORE TIMES AND
C                             THE STEP WAS TRIED AGAIN SUCCESSFULLY.
C                         TERMINAL ERROR
C                           IER = 132, IMPLIES THE INTEGRATION WAS
C                             HALTED AFTER FAILING TO PASS THE ERROR
C                             TEST EVEN AFTER REDUCING H BY A FACTOR
C                             OF 1.0E10 FROM ITS INITIAL VALUE.
C                             SEE REMARKS.
C                           IER = 133, IMPLIES THE INTEGRATION WAS
C                             HALTED AFTER FAILING TO ACHIEVE
C                             CORRECTOR CONVERGENCE EVEN AFTER
C                             REDUCING H BY A FACTOR OF 1.0E10 FROM
C                             ITS INITIAL VALUE. SEE REMARKS.
C                           IER = 134, IMPLIES THAT AFTER SOME INITIAL
C                             SUCCESS, THE INTEGRATION WAS HALTED EITHER
C                             BY REPEATED ERROR TEST FAILURES OR BY
C                             A TEST ON TOL. SEE REMARKS.
C                           IER = 135, IMPLIES THAT ONE OF THE INPUT
C                             PARAMETERS N,X,H,XEND,TOL,METH,MITER, OR
C                             INDEX WAS SPECIFIED INCORRECTLY.
C                           IER = 136, IMPLIES THAT INDEX HAD A VALUE
C                             OF -1 ON INPUT, BUT THE DESIRED CHANGES
C                             OF PARAMETERS WERE NOT IMPLEMENTED
C                             BECAUSE XEND WAS NOT BEYOND X.
C                             INTERPOLATION TO X = XEND WAS PERFORMED.
C                             TO TRY AGAIN, SIMPLY CALL AGAIN WITH
C                             INDEX EQUAL TO -1 AND A NEW VALUE FOR
C                             XEND.
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - DGRCS,DGRIN,DGRPS,DGRST,LUDATF,LUELMF,LEQT1B,
C                           UERTST,UGETIO
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   REMARKS  1.  THE EXTERNAL SUBROUTINE FCNJ IS USED ONLY WHEN
C                INPUT PARAMETER MITER IS EQUAL TO 1 OR -1. OTHERWISE,
C                A DUMMY FUNCTION CAN BE USED. THE DUMMY SUBROUTINE
C                SHOULD BE OF THE FOLLOWING FORM
C                  SUBROUTINE FCNJ (N,X,Y,PD)
C                  INTEGER N
C                  REAL Y(N),PD(N,N),X
C                  RETURN
C                  END
C            2.  AFTER THE INITIAL CALL, IF A NORMAL RETURN OCCURRED
C                (IER=0) AND A NORMAL CONTINUATION IS DESIRED, SIMPLY
C                RESET XEND AND CALL DGEAR AGAIN. ALL OTHER
C                PARAMETERS WILL BE READY FOR THE NEXT CALL. A CHANGE
C                OF PARAMETERS WITH INDEX EQUAL TO -1 CAN BE MADE
C                AFTER EITHER A SUCCESSFUL OR AN UNSUCCESSFUL RETURN.
C            3.  THE COMMON BLOCKS /DBAND/ AND /GEAR/ NEED TO BE
C                PRESERVED BETWEEN CALLS TO DGEAR. IF IT IS NECESSARY
C                FOR THE COMMON BLOCKS TO EXIST IN THE CALLING PROGRAM
C                THE FOLLOWING STATEMENTS SHOULD BE INCLUDED
C                  COMMON  /DBAND/ NLC,NUC
C                  COMMON  /GEAR/ DUMMY(48),SDUMMY(4),IDUMMY(38)
C                WHERE DUMMY, SDUMMY, AND IDUMMY ARE VARIABLE NAMES NOT
C                USED ELSEWHERE IN THE CALLING PROGRAM.  (FOR DOUBLE
C                PRECISION DUMMY IS TYPE DOUBLE AND SDUMMY IS TYPE REAL)
C            4.  THE CHOICE OF VALUES FOR METH AND MITER MAY REQUIRE
C                SOME EXPERIMENTATION, AND ALSO SOME CONSIDERATION OF
C                THE NATURE OF THE PROBLEM AND OF STORAGE REQUIREMENTS.
C                THE PRIME CONSIDERATION IS STIFFNESS. IF
C                THE PROBLEM IS NOT STIFF, THE BEST CHOICE IS PROBABLY
C                METH = 1 WITH MITER = 0. IF THE PROBLEM IS STIFF TO A
C                SIGNIFICANT DEGREE, THEN METH SHOULD BE 2 AND MITER
C                SHOULD BE 1,2,-1,-2 OR 3. IF THE USER HAS NO KNOWLEDGE
C                OF THE INHERENT TIME CONSTANTS OF THE PROBLEM, WITH
C                WHICH TO PREDICT ITS STIFFNESS, ONE WAY TO DETERMINE
C                THIS IS TO TRY METH = 1 AND MITER = 0 FIRST, AND LOOK
C                AT THE BEHAVIOR OF THE SOLUTION COMPUTED AND THE STEP
C                SIZES USED. IF THE TYPICAL VALUES OF H ARE MUCH
C                SMALLER THAN THE SOLUTION BEHAVIOR WOULD SEEM TO
C                REQUIRE (THAT IS, MORE THAN 100 STEPS ARE TAKEN OVER
C                AN INTERVAL IN WHICH THE SOLUTIONS CHANGE BY LESS
C                THAN ONE PERCENT), THEN THE PROBLEM IS PROBABLY STIFF
C                AND THE DEGREE OF STIFFNESS CAN BE ESTIMATED FROM THE
C                VALUES OF H USED AND THE SMOOTHNESS OF THE SOLUTION.
C                IF THE DEGREE OF STIFFNESS IS ONLY SLIGHT, IT MAY BE
C                THAT METH=1 IS MORE EFFICIENT THAN METH=2.
C                EXPERIMENTATION WOULD BE REQUIRED TO DETERMINE THIS.
C                REGARDLESS OF METH, THE LEAST EFFECTIVE VALUE OF
C                MITER IS 0, AND THE MOST EFFECTIVE IS 1,-1,2,OR -2.
C                MITER = 3 IS GENERALLY SOMEWHERE IN BETWEEN. SINCE
C                THE STORAGE REQUIREMENTS GO UP IN THE SAME ORDER AS
C                EFFECTIVENESS, TRADE-OFF CONSIDERATIONS ARE
C                NECESSARY. FOR REASONS OF ACCURACY AND SPEED, THE
C                CHOICE OF ABS(MITER)=1 IS GENERALLY PREFERRED TO
C                ABS(MITER)=2, UNLESS THE SYSTEM IS FAIRLY COMPLICATED
C                (AND FCNJ IS THUS NOT FEASIBLE TO CODE). THE
C                ACCURACY OF THE FCNJ CALCULATION CAN BE CHECKED BY
C                COMPARISON OF THE JACOBIAN WITH THAT GENERATED WITH
C                ABS(MITER)=2. IF THE JACOBIAN MATRIX IS SIGNIFICANTLY
C                DIAGONALLY DOMINANT, THEN THE OPTION MITER = 3 IS
C                LIKELY TO BE NEARLY AS EFFECTIVE AS ABS(MITER)=1 OR 2,
C                AND WILL SAVE CONSIDERABLE STORAGE AND RUN TIME.
C                IT IS POSSIBLE, AND POTENTIALLY QUITE DESIRABLE, TO
C                USE DIFFERENT VALUES OF METH AND MITER IN DIFFERENT
C                SUBINTERVALS OF THE PROBLEM. FOR EXAMPLE, IF THE
C                PROBLEM IS NON-STIFF INITIALLY AND STIFF LATER,
C                METH = 1 AND MITER = 0 MIGHT BE SET INITIALLY, AND
C                METH = 2 AND MITER = 1 LATER.
C            5.  THE INITIAL VALUE OF THE STEP SIZE, H, SHOULD BE
C                CHOSEN CONSIDERABLY SMALLER THAN THE AVERAGE VALUE
C                EXPECTED FOR THE PROBLEM, AS THE FIRST-ORDER METHOD
C                WITH WHICH DGEAR BEGINS IS NOT GENERALLY THE MOST
C                EFFICIENT ONE. HOWEVER, FOR THE FIRST STEP, AS FOR
C                EVERY STEP, DGEAR TESTS FOR THE POSSIBILITY THAT
C                THE STEP SIZE WAS TOO LARGE TO PASS THE ERROR TEST
C                (BASED ON TOL), AND IF SO ADJUSTS THE STEP SIZE
C                DOWN AUTOMATICALLY. THIS DOWNWARD ADJUSTMENT, IF
C                ANY, IS NOTED BY IER HAVING THE VALUES 66 OR 67,
C                AND SUBSEQUENT RUNS ON THE SAME OR SIMILAR PROBLEM
C                SHOULD BE STARTED WITH AN APPROPRIATELY SMALLER
C                VALUE OF H.
C            6.  SOME OF THE VALUES OF INTEREST LOCATED IN THE
C                COMMON BLOCK /GEAR/ ARE
C                A. HUSED, THE STEP SIZE H LAST USED SUCCESSFULLY
C                   (DUMMY(8))
C                B. NQUSED, THE ORDER LAST USED SUCCESSFULLY
C                   (IDUMMY(6))
C                C. NSTEP, THE CUMULATIVE NUMBER OF STEPS TAKEN
C                   (IDUMMY(7))
C                D. NFE, THE CUMULATIVE NUMBER OF FCN EVALUATIONS
C                   (IDUMMY(8))
C                E. NJE, THE CUMULATIVE NUMBER OF JACOBIAN
C                   EVALUATIONS, AND HENCE ALSO OF MATRIX LU
C                   DECOMPOSITIONS (IDUMMY(9))
C            7.  THE NORMAL USAGE OF DGEAR MAY BE SUMMARIZED AS FOLLOWS
C                A. SET THE INITIAL VALUES IN Y.
C                B. SET N, X, H, TOL, METH, AND MITER.
C                C. SET XEND TO THE FIRST OUTPUT POINT, AND INDEX TO 1.
C                D. CALL DGEAR
C                E. EXIT IF IER IS GREATER THAN 128.
C                F. OTHERWISE, DO DESIRED OUTPUT OF Y.
C                G. EXIT IF THE PROBLEM IS FINISHED.
C                H. OTHERWISE, RESET XEND TO THE NEXT OUTPUT POINT, AND
C                   RETURN TO STEP D.
C            8.  THE ERROR WHICH IS CONTROLLED BY WAY OF THE PARAMETER
C                TOL IS AN ESTIMATE OF THE LOCAL TRUNCATION ERROR, THAT
C                IS, THE ERROR COMMITTED ON TAKING A SINGLE STEP WITH
C                THE METHOD, STARTING WITH DATA REGARDED AS EXACT. THIS
C                IS TO BE DISTINGUISHED FROM THE GLOBAL TRUNCATION
C                ERROR, WHICH IS THE ERROR IN ANY GIVEN COMPUTED VALUE
C                OF Y(X) AS A RESULT OF THE LOCAL TRUNCATION ERRORS
C                FROM ALL STEPS TAKEN TO OBTAIN Y(X). THE LATTER ERROR
C                ACCUMULATES IN A NON-TRIVIAL WAY FROM THE LOCAL
C                ERRORS, AND IS NEITHER ESTIMATED NOR CONTROLLED BY
C                THE ROUTINE. SINCE IT IS USUALLY THE GLOBAL ERROR THAT
C                A USER WANTS TO HAVE UNDER CONTROL, SOME
C                EXPERIMENTATION MAY BE NECESSARY TO GET THE RIGHT
C                VALUE OF TOL TO ACHIEVE THE USERS NEEDS. IF THE
C                PROBLEM IS MATHEMATICALLY STABLE, AND THE METHOD USED
C                IS APPROPRIATELY STABLE, THEN THE GLOBAL ERROR AT A
C                GIVEN X SHOULD VARY SMOOTHLY WITH TOL IN A MONOTONE
C                INCREASING MANNER.
C            9.  IF THE ROUTINE RETURNS WITH IER VALUES OF 132, 133,
C                OR 134, THE USER SHOULD CHECK TO SEE IF TOO MUCH
C                ACCURACY IS BEING REQUIRED. THE USER MAY WISH TO
C                SET TOL TO A LARGER VALUE AND CONTINUE. ANOTHER
C                POSSIBLE CAUSE OF THESE ERROR CONDITIONS IS AN
C                ERROR IN THE CODING OF THE EXTERNAL FUNCTIONS FCN
C                OR FCNJ. IF NO ERRORS ARE FOUND, IT MAY BE NECESSARY
C                TO MONITOR INTERMEDIATE QUANTITIES GENERATED BY THE
C                ROUTINE. THESE QUANTITIES ARE STORED IN THE WORK VECTOR
C                WK AND INDEXED BY SPECIFIC ELEMENTS IN THE COMMON BLOCK
C                /GEAR/. IF IER IS 132 OR 134, THE COMPONENTS CAUSING
C                THE ERROR TEST FAILURE CAN BE IDENTIFIED FROM LARGE
C                VALUES OF THE QUANTITY
C                  WK(IDUMMY(11)+I)/WK(I), FOR I=1,...,N.
C                ONE CAUSE OF THIS MAY BE A VERY SMALL BUT NONZERO
C                INITIAL VALUE OF ABS(Y(I)).
C                IF IER IS 133, SEVERAL POSSIBILITIES EXIST.
C                IT MAY BE INSTRUCTIVE TO TRY DIFFERENT VALUES OF MITER.
C                ALTERNATIVELY, THE USER MIGHT MONITOR SUCCESSIVE
C                CORRECTOR ITERATES CONTAINED IN WK(IDUMMY(12)+I), FOR
C                I=1,...,N. ANOTHER POSSIBILITY MIGHT BE TO MONITOR
C                THE JACOBIAN MATRIX, IF ONE IS USED, STORED, BY
C                COLUMN, IN WK(IDUMMY(10)+I), FOR I=1,...,N*N IF
C                ABS(MITER) IS EQUAL TO 1 OR 2, OR FOR I=1,...,N IF
C                MITER IS EQUAL TO 3.
C
C   COPYRIGHT           - 1984 BY IMSL, INC.  ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE.  NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DGEAR  (N,FCN,FCNJ,X,H,Y,XEND,TOL,METH,MITER,INDEX,
     1                   IWK,WK,IER)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            N,METH,MITER,INDEX,IWK(*),IER
      DOUBLE PRECISION   X,H,Y(N),XEND,TOL,WK(*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            NERROR,NSAVE1,NSAVE2,NPW,NY,NC,MFC,KFLAG,
     1                   JSTART,NSQ,NQUSED,NSTEP,NFE,NJE,I,N0,NHCUT,KGO,
     2                   JER,KER,NN,NEQUIL,IDUMMY(21),NLC,NUC
Cws   REAL               SDUMMY(4) 
      DOUBLE PRECISION   SDUMMY(4)
      DOUBLE PRECISION   T,HH,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,TOUTP,
     1                   AYI,D,DN,SEPS,DUMMY(39)
      EXTERNAL           FCN,FCNJ
      COMMON /DBAND/     NLC,NUC
      COMMON /GEAR/      T,HH,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,DUMMY,
     1                   TOUTP,SDUMMY,NC,MFC,KFLAG,JSTART,NSQ,NQUSED,
     2                   NSTEP,NFE,NJE,NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,
     3                   NY,IDUMMY,N0,NHCUT
      DATA               SEPS/.2220446050D-15/
C                                  FIRST EXECUTABLE STATEMENT

      IF (MITER.GE.0) NLC = -1
      KER = 0
      JER = 0
      UROUND = SEPS
C                                  COMPUTE WORK VECTOR INDICIES
      NERROR = N
      NSAVE1 = NERROR+N
      NSAVE2 = NSAVE1+N
      NY = NSAVE2+N
      IF (METH.EQ.1) NEQUIL = NY+13*N
      IF (METH.EQ.2) NEQUIL = NY+6*N
      NPW = NEQUIL + N
      IF (MITER.EQ.0.OR.MITER.EQ.3) NPW = NEQUIL
      MFC = 10*METH+IABS(MITER)
C                                  CHECK FOR INCORRECT INPUT PARAMETERS
C
      IF (MITER.LT.-2.OR.MITER.GT.3) GO TO 85
      IF (METH.NE.1.AND.METH.NE.2) GO TO 85
      IF (TOL.LE.0.D0) GO TO 85
      IF (N.LE.0) GO TO 85
      IF ((X-XEND)*H.GE.0.D0) GO TO 85
      IF (INDEX.EQ.0) GO TO 10
      IF (INDEX.EQ.2) GO TO 15
      IF (INDEX.EQ.-1) GO TO 20
      IF (INDEX.EQ.3) GO TO 25
      IF (INDEX.NE.1) GO TO 85
C                                  IF INITIAL VALUES OF YMAX OTHER THAN
C                                    THOSE SET BELOW ARE DESIRED, THEY
C                                    SHOULD BE SET HERE. ALL YMAX(I)
C                                    MUST BE POSITIVE. IF VALUES FOR
C                                    HMIN OR HMAX, THE BOUNDS ON
C                                    DABS(HH), OTHER THAN THOSE BELOW
C                                    ARE DESIRED, THEY SHOULD BE SET
C                                    BELOW.
      DO 5 I=1,N
         WK(I) = DABS(Y(I))
         IF (WK(I).EQ.0.D0) WK(I) = 1.D0
         WK(NY+I) = Y(I)
    5 CONTINUE
      NC = N
      T = X
      HH = H
      IF ((T+HH).EQ.T) KER = 33
      HMIN = DABS(H)
      HMAX = DABS(X-XEND)*10.D0
      EPSC = TOL
      JSTART = 0
      N0 = N
      NSQ = N0*N0
      EPSJ = DSQRT(UROUND)
      NHCUT = 0
      DUMMY(2) = 1.0D0
      DUMMY(14) = 1.0D0
      GO TO 30
C                                  TOUTP IS THE PREVIOUS VALUE OF XEND
C                                    FOR USE IN HMAX.
   10 HMAX = DABS(XEND-TOUTP)*10.D0
      GO TO 45
C
   15 HMAX = DABS(XEND-TOUTP)*10.D0
      IF ((T-XEND)*HH.GE.0.D0) GO TO 95
      GO TO 50
C
   20 IF ((T-XEND)*HH.GE.0.D0) GO TO 90
      JSTART = -1
      NC = N
      EPSC = TOL
C
   25 IF ((T+HH).EQ.T) KER = 33
C
   30 NN = N0
      CALL DGRST (FCN,FCNJ,WK(NY+1),WK,WK(NERROR+1),WK(NSAVE1+1),
     1 WK(NSAVE2+1),WK(NPW+1),WK(NEQUIL+1),IWK,NN)
C
      KGO = 1-KFLAG
      GO TO (35,55,70,80), KGO
C                                  KFLAG = 0, -1, -2, -3
   35 CONTINUE
C                                  NORMAL RETURN FROM INTEGRATOR. THE
C                                    WEIGHTS YMAX(I) ARE UPDATED. IF
C                                    DIFFERENT VALUES ARE DESIRED, THEY
C                                    SHOULD BE SET HERE. A TEST IS MADE
C                                    FOR TOL BEING TOO SMALL FOR THE
C                                    MACHINE PRECISION. ANY OTHER TESTS
C                                    OR CALCULATIONS THAT ARE REQUIRED
C                                    AFTER EVERY STEP SHOULD BE
C                                    INSERTED HERE. IF INDEX = 3, Y IS
C                                    SET TO THE CURRENT SOLUTION ON
C                                    RETURN. IF INDEX = 2, HH IS
C                                    CONTROLLED TO HIT XEND (WITHIN
C                                    ROUNDOFF ERROR), AND THEN THE
C                                    CURRENT SOLUTION IS PUT IN Y ON
C                                    RETURN. FOR ANY OTHER VALUE OF
C                                    INDEX, CONTROL RETURNS TO THE
C                                    INTEGRATOR UNLESS XEND HAS BEEN
C                                    REACHED. THEN INTERPOLATED VALUES
C                                    OF THE SOLUTION ARE COMPUTED AND
C                                    STORED IN Y ON RETURN.
C                                    IF INTERPOLATION IS NOT
C                                    DESIRED, THE CALL TO DGRIN SHOULD
C                                    BE REMOVED AND CONTROL TRANSFERRED
C                                    TO STATEMENT 95 INSTEAD OF 105.
      D = 0.D0
      DO 40 I=1,N
         AYI = DABS(WK(NY+I))
         WK(I) = DMAX1(WK(I),AYI)
   40 D = D+(AYI/WK(I))**2
      D = D*(UROUND/TOL)**2
      DN = N
      IF (D.GT.DN) GO TO 75
      IF (INDEX.EQ.3) GO TO 95
      IF (INDEX.EQ.2) GO TO 50
   45 IF ((T-XEND)*HH.LT.0.D0) GO TO 25
      NN = N0
      CALL DGRIN (XEND,WK(NY+1),NN,Y)
      X = XEND
      GO TO 105
   50 IF (((T+HH)-XEND)*HH.LE.0.D0) GO TO 25
      IF (DABS(T-XEND).LE.UROUND*DMAX1(10.D0*DABS(T),HMAX)) GO TO 95
      IF ((T-XEND)*HH.GE.0.D0) GO TO 95
      HH = (XEND-T)*(1.D0-4.D0*UROUND)
      JSTART = -1
      GO TO 25
C                                  ON AN ERROR RETURN FROM INTEGRATOR,
C                                    AN IMMEDIATE RETURN OCCURS IF
C                                    KFLAG = -2, AND RECOVERY ATTEMPTS
C                                    ARE MADE OTHERWISE. TO RECOVER, HH
C                                    AND HMIN ARE REDUCED BY A FACTOR
C                                    OF .1 UP TO 10 TIMES BEFORE GIVING
C                                    UP.
   55 JER = 66
   60 IF (NHCUT.EQ.10) GO TO 65
      NHCUT = NHCUT+1
      HMIN = HMIN*.1D0
      HH = HH*.1D0
      JSTART = -1
      GO TO 25
C
   65 IF (JER.EQ.66) JER = 132
      IF (JER.EQ.67) JER = 133
      GO TO 95
C
   70 JER = 134
      GO TO 95
C
   75 JER = 134
      KFLAG = -2
      GO TO 95
C
   80 JER = 67
      GO TO 60
C
   85 JER = 135
      GO TO 110
C
   90 JER = 136
      NN = N0
      CALL DGRIN (XEND,WK(NY+1),NN,Y)
      X = XEND
      GO TO 110
C
   95 X = T
      DO 100 I=1,N
  100 Y(I) = WK(NY+I)
  105 IF (JER.LT.128) INDEX = KFLAG
      TOUTP = X
      IF (KFLAG.EQ.0) H = HUSED
      IF (KFLAG.NE.0) H = HH
  110 IER = MAX0(KER,JER)
 9000 CONTINUE
      IF (KER.NE.0.AND.JER.LT.128) then
c        write(*,*) (y(i),i=1,n)
         CALL UERTST (KER,'DGEAR ')
      endif
      IF (JER.NE.0) then
c         write(*,*) (y(i),i=1,n)
         CALL UERTST (JER,'DGEAR ')
      endif
 9005 RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JANUARY 1, 1978
C
C   PURPOSE             - NUCLEUS CALLED ONLY BY IMSL SUBROUTINE DGEAR
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - NONE REQUIRED
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DGRCS  (METH,NQ,EL,TQ,MAXDER)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            METH,NQ,MAXDER
      DOUBLE PRECISION   TQ(*)
      DOUBLE PRECISION   EL(*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            K
      DOUBLE PRECISION   PERTST(12,2,3)
      DATA               PERTST/1.,1.,2.,1.,.3158,.7407D-1,
     1                   .1391D-1,.2182D-2,.2945D-3,.3492D-4,
     2                   .3692D-5,.3524D-6,1.,1.,.5,.1667,
     3                   .4167D-1,7*1.,2.,12.,24.,37.89,
     4                   53.33,70.08,87.97,106.9,126.7,
     5                   147.4,168.8,191.0,2.0,4.5,7.333,
     6                   10.42,13.7,7*1.,12.0,24.0,37.89,
     7                   53.33,70.08,87.97,106.9,126.7,
     8                   147.4,168.8,191.0,1.,3.0,6.0,
     9                   9.167,12.5,8*1./

C                                  THE FOLLOWING COEFFICIENTS SHOULD BE
C                                    DEFINED TO MACHINE ACCURACY. FOR A
C                                    GIVEN ORDER NQ, THEY CAN BE
C                                    CALCULATED BY USE OF THE
C                                    GENERATING POLYNOMIAL L(T), WHOSE
C                                    COEFFICIENTS ARE EL(I).. L(T) =
C                                    EL(1) + EL(2)*T + ... +
C                                    EL(NQ+1)*T**NQ. FOR THE IMPLICIT
C                                    ADAMS METHODS, L(T) IS GIVEN BY
C                                    DL/DT = (T+1)*(T+2)* ...
C                                    *(T+NQ-1)/K, L(-1) = 0, WHERE K =
C                                    FACTORIAL(NQ-1). FOR THE GEAR
C                                    METHODS, L(T) = (T+1)*(T+2)* ...
C                                    *(T+NQ)/K, WHERE K =
C                                    FACTORIAL(NQ)*(1 + 1/2 + ... +
C                                    1/NQ). THE ORDER IN WHICH THE
C                                    GROUPS APPEAR BELOW IS.. IMPLICIT
C                                    ADAMS METHODS OF ORDERS 1 TO 12,
C                                    BACKWARD DIFFERENTIATION METHODS
C                                    OF ORDERS 1 TO 5.
      if (meth.eq.1) then
        maxder=12
        if     (nq.eq.1) then
          EL(1) = 1.0D0
        elseif (nq.eq.2) then
          EL(1) = 0.5D0
          EL(3) = 0.5D0
        elseif (nq.eq.3) then
          EL(1) = 4.166666666666667D-01
          EL(3) = 0.75D0
          EL(4) = 1.666666666666667D-01
        elseif (nq.eq.4) then
          EL(1) = 0.375D0
          EL(3) = 9.166666666666667D-01
          EL(4) = 3.333333333333333D-01
          EL(5) = 4.166666666666667D-02
        elseif (nq.eq.5) then
          EL(1) = 3.486111111111111D-01
          EL(3) = 1.041666666666667D0
          EL(4) = 4.861111111111111D-01
          EL(5) = 1.041666666666667D-01
          EL(6) = 8.333333333333333D-03
        elseif (nq.eq.6) then
          EL(1) = 3.298611111111111D-01
          EL(3) = 1.141666666666667D+00
          EL(4) = 0.625D+00
          EL(5) = 1.770833333333333D-01
          EL(6) = 0.025D+00
          EL(7) = 1.388888888888889D-03
        elseif (nq.eq.7) then
          EL(1) = 3.155919312169312D-01
          EL(3) = 1.225D+00
          EL(4) = 7.518518518518519D-01
          EL(5) = 2.552083333333333D-01
          EL(6) = 4.861111111111111D-02
          EL(7) = 4.861111111111111D-03
          EL(8) = 1.984126984126984D-04
        elseif (nq.eq.8) then
          EL(1) = 3.042245370370370D-01
          EL(3) = 1.296428571428571D+00
          EL(4) = 8.685185185185185D-01
          EL(5) = 3.357638888888889D-01
          EL(6) = 7.777777777777778D-02
          EL(7) = 1.064814814814815D-02
          EL(8) = 7.936507936507937D-04
          EL(9) = 2.480158730158730D-05
        elseif (nq.eq.9) then
          EL(1) = 2.948680004409171D-01
          EL(3) = 1.358928571428571D+00
          EL(4) = 9.765542328042328D-01
          EL(5) = 4.171875D-01
          EL(6) = 1.113541666666667D-01
          EL(7) = 0.01875D+00
          EL(8) = 1.934523809523810D-03
          EL(9) = 1.116071428571429D-04
          EL(10)= 2.755731922398589D-06
        elseif (nq.eq.10) then
          EL(1) = 2.869754464285714D-01
          EL(3) = 1.414484126984127D+00
          EL(4) = 1.077215608465609D+00
          EL(5) = 4.985670194003527D-01
          EL(6) = 1.484375D-01
          EL(7) = 2.906057098765432D-02
          EL(8) = 3.720238095238095D-03
          EL(9) = 2.996858465608466D-04
          EL(10)= 1.377865961199295D-05
          EL(11)= 2.755731922398589D-07
        elseif (nq.eq.11) then
          EL(1) = 2.801895964439367D-01
          EL(3) = 1.464484126984127D+00
          EL(4) = 1.171514550264550D+00
          EL(5) = 5.793581900352734D-01
          EL(6) = 1.883228615520282D-01
          EL(7) = 4.143036265432099D-02
          EL(8) = 6.211144179894180D-03
          EL(9) = 6.252066798941799D-04
          EL(10)= 4.041740152851264D-05
          EL(11)= 1.515652557319224D-06
          EL(12)= 2.505210838544172D-08
        elseif (nq.eq.12) then
          EL(1) = 2.742655400315991D-01
          EL(3) = 1.509938672438672D+00
          EL(4) = 1.260271164021164D+00
          EL(5) = 6.592341820987654D-01
          EL(6) = 2.304580026455027D-01
          EL(7) = 5.569724610523222D-02
          EL(8) = 9.439484126984127D-03
          EL(9) = 1.119274966931217D-03
          EL(10)= 9.093915343915344D-05
          EL(11)= 4.822530864197531D-06
          EL(12)= 1.503126503126503D-07
          EL(13)= 2.087675698786810D-09
        endif
      elseif (meth.eq.2) then
        maxder=5
        if     (nq.eq.1) then
          EL(1) = 1.0D+00
        elseif (nq.eq.2) then
          EL(1) = 6.666666666666667D-01
          EL(3) = 3.333333333333333D-01
        elseif (nq.eq.3) then
          EL(1) = 5.454545454545455D-01
          EL(3) = EL(1)
          EL(4) = 9.090909090909091D-02
        elseif (nq.eq.4) then
          EL(1) = 0.48D+00
          EL(3) = 0.7D+00
          EL(4) = 0.2D+00
          EL(5) = 0.02D+00
        elseif (nq.eq.5) then
          EL(1) = 4.379562043795620D-01
          EL(3) = 8.211678832116788D-01
          EL(4) = 3.102189781021898D-01
          EL(5) = 5.474452554744526D-02
          EL(6) = 3.649635036496350D-03
        endif
      endif
C
  100 DO 105 K=1,3
         TQ(K) = PERTST(NQ,METH,K)
  105 CONTINUE
      TQ(4) = .5D0*TQ(2)/(NQ+2)
      RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JANUARY 1, 1978
C
C   PURPOSE             - NUCLEUS CALLED ONLY BY IMSL SUBROUTINE DGEAR
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - NONE REQUIRED
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DGRIN  (TOUT,Y,N0,Y0)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            N0
      DOUBLE PRECISION   TOUT,Y0(N0),Y(N0,*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            NC,MFC,KFLAG,I,L,J,JSTART,NSQ,NQUSED,NSTEP,
     1                   NFE,NJE,NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     2                   IDUMMY(23)
      DOUBLE PRECISION   SDUMMY(4)
CWS   REAL               SDUMMY(4)
      DOUBLE PRECISION   T,H,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,S,S1,
     1                   DUMMY(40)
      COMMON /GEAR/      T,H,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,DUMMY,
     1                   SDUMMY,NC,MFC,KFLAG,JSTART,NSQ,NQUSED,NSTEP,
     2                   NFE,NJE,NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     3                   IDUMMY
C                                  FIRST EXECUTABLE STATEMENT
      DO 5 I = 1,NC
         Y0(I) = Y(I,1)
    5 CONTINUE
C                                  THIS SUBROUTINE COMPUTES INTERPOLATED
C                                    VALUES OF THE DEPENDENT VARIABLE
C                                    Y AND STORES THEM IN Y0. THE
C                                    INTERPOLATION IS TO THE
C                                    POINT T = TOUT, AND USES THE
C                                    NORDSIECK HISTORY ARRAY Y, AS
C                                    FOLLOWS..
C                                               NQ
C                                    Y0(I)  =  SUM  Y(I,J+1)*S**J ,
C                                              J=0
C                                    WHERE S = -(T-TOUT)/H.
      L = JSTART + 1
      S = (TOUT - T)/H
      S1 = 1.0D0
      DO 15 J = 2,L
         S1 = S1*S
         DO 10 I = 1,NC
            Y0(I) = Y0(I) + S1*Y(I,J)
   10    CONTINUE
   15 CONTINUE
      RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - NOVEMBER 1, 1984
C
C   PURPOSE             - NUCLEUS CALLED ONLY BY IMSL SUBROUTINE DGEAR
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - LUDATF,LEQT1B,UERTST,UGETIO
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1984 BY IMSL, INC.  ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE.  NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DGRPS (FCN,FCNJ,Y,N0,CON,MITER,YMAX,SAVE1,SAVE2,PW,
     *                   EQUIL,IPIV,IER)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            N0,MITER,IPIV(*),IER
      DOUBLE PRECISION   Y(N0,*),CON,YMAX(*),SAVE1(*),SAVE2(*),PW(*),
     *                   EQUIL(*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
C
      integer*4            NC,MFC,KFLAG,JSTART,NQUSED,NSTEP,NFE,NJE,NPW,
     *                   NSQ,I,J1,J,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     *                   IDUMMY(23),NLIM,II,IJ,LIM1,LIM2,NB,NLC,NUC,NWK
      DOUBLE PRECISION   SDUMMY(4)
      DOUBLE PRECISION   T,H,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,D,R0,YJ,R,
     *                   D1,D2,WA,DUMMY(40)
      EXTERNAL           FCN,FCNJ
      COMMON /DBAND/     NLC,NUC
      COMMON /GEAR/      T,H,HMIN,HMAX,EPSC,UROUND,EPSJ,HUSED,DUMMY,
     *                   SDUMMY,NC,MFC,KFLAG,JSTART,NSQ,NQUSED,NSTEP,
     *                   NFE,NJE,NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     *                   IDUMMY
C                                  THIS ROUTINE IS CALLED BY DGRST TO
C                                    COMPUTE AND PROCESS THE MATRIX P =
C                                    I - H*EL(1)*J , WHERE J IS AN
C                                    APPROXIMATION TO THE JACOBIAN. J
C                                    IS COMPUTED, EITHER BY THE USER-
C                                    SUPPLIED ROUTINE FCNJ IF MITER =
C                                    1, OR BY FINITE DIFFERENCING IF
C                                    MITER = 2. J IS STORED IN PW AND
C                                    REPLACED BY P, USING CON =
C                                    -H*EL(1). THEN P IS SUBJECTED TO
C                                    LU DECOMPOSITION IN PREPARATION
C                                    FOR LATER SOLUTION OF LINEAR
C                                    SYSTEMS WITH P AS COEFFICIENT
C                                    MATRIX. IN ADDITION TO VARIABLES
C                                    DESCRIBED PREVIOUSLY,
C                                    COMMUNICATION WITH DGRPS USES THE
C                                    FOLLOWING EPSJ = DSQRT(UROUND),
C                                    USED IN THE NUMERICAL JACOBIAN
C                                    INCREMENTS.
C
      if (nlc.ne.-1) then
C                                  BANDED JACOBIAN CASE
        NB = NLC+NUC+1
        NWK = NB*N0+1
C                                  MITER = 1
        if (miter.ne.2) then
          NLIM = NB*N0
          DO 5 I=1,NLIM
            PW(I) = 0.0D0
    5     CONTINUE
          CALL FCNJ(NC,T,Y,PW)
          DO 10 I=1,NLIM
            PW(I) = PW(I)*CON
   10     CONTINUE

        else
C                                  MITER = 2
          D = 0.0D0
          DO 20 I=1,NC
   20     D = D+SAVE2(I)**2
          R0 = DABS(H)*DSQRT(D)*1.0D+03*UROUND
          DO 30 J=1,NC
            YJ = Y(J,1)
            R = EPSJ*YMAX(J)
            R = DMAX1(R,R0)
            Y(J,1) = Y(J,1)+R
            D = CON/R
            CALL FCN(NC,T,Y,SAVE1)
            lim1=j-nuc
            if (lim1.lt.1) lim1=1
c            LIM1 = MAX0(1,J-NUC)
            LIM2 = MIN0(N0,J+NLC)
            DO 25 I=LIM1,LIM2
              IJ = (J-I+NLC)*N0+I
              PW(IJ) = (SAVE1(I)-SAVE2(I))*D
   25       CONTINUE
            Y(J,1) = YJ
   30     CONTINUE
        endif
C                                  ADD IDENTITY MATRIX.
   35   DO 40 I=1,NC
          II = NLC*N0+I
          PW(II) = PW(II)+1.0D0
   40   CONTINUE
C                                  DO LU DECOMPOSITION ON P
C
        CALL LEQT1B(PW,NC,NLC,NUC,N0,EQUIL,1,N0,1,PW(NWK),IER)

      else
C                                  FULL JACOBIAN CASE
        if (miter.ne.2) then
C                                  MITER = 1
          CALL FCNJ(NC,T,Y,PW)
          DO 50 I=1,NSQ
   50     PW(I) = PW(I)*CON
        else
C                                  MITER = 2
   55     D = 0.0D0
          DO 60 I=1,NC
   60     D = D+SAVE2(I)**2
          R0 = DABS(H)*DSQRT(D)*1.0D+03*UROUND
          J1 = 0
          DO 70 J=1,NC
            YJ = Y(J,1)
            R = EPSJ*YMAX(J)
            R = DMAX1(R,R0)
            Y(J,1) = Y(J,1)+R
            D = CON/R
            CALL FCN(NC,T,Y,SAVE1)
            DO 65 I=1,NC
   65       PW(I+J1) = (SAVE1(I)-SAVE2(I))*D
            Y(J,1) = YJ
            J1 = J1+N0
   70     CONTINUE
        endif

C                                  ADD IDENTITY MATRIX.
        J = 1
        DO 80 I=1,NC
          PW(J) = PW(J)+1.0D0
          J = J+(N0+1)
   80   CONTINUE
C                                  DO LU DECOMPOSITION ON P.
C
        CALL LUDATF(PW,PW,NC,N0,0,D1,D2,IPIV,EQUIL,WA,IER)

      endif

      RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JUNE 1, 1982
C
C   PURPOSE             - NUCLEUS CALLED ONLY BY IMSL SUBROUTINE DGEAR
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - DGRCS,DGRPS,LUDATF,LUELMF,LEQT1B,UERTST,
C                           UGETIO
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1982 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DGRST  (FCN,FCNJ,Y,YMAX,ERROR,SAVE1,SAVE2,PW,EQUIL,
     1                   IPIV,N0)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            IPIV(*),N0
      DOUBLE PRECISION   Y(N0,*),YMAX(*),ERROR(*),SAVE1(*),SAVE2(*),
     1                   PW(*),EQUIL(*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            N,MF,KFLAG,JSTART,NQUSED,NSTEP,NFE,NJE,NSQ,
     1                   I,METH,MITER,NQ,L,IDOUB,MFOLD,NOLD,IRET,MEO,
     2                   MIO,IWEVAL,MAXDER,LMAX,IREDO,J,NSTEPJ,J1,J2,
     3                   M,IER,NEWQ,NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     4                   MITER1,IDUMMY(2),NLC,NUC,NWK,JER
CWS   REAL               TQ(4)
      DOUBLE PRECISION   TQ(4) 
      DOUBLE PRECISION   T,H,HMIN,HMAX,EPS,UROUND,HUSED,EL(13),OLDL0,
     1                   TOLD,RMAX,RC,CRATE,EPSOLD,HOLD,FN,EDN,E,EUP,
     2                   BND,RH,R1,CON,R,HL0,R0,D,PHL0,PR3,D1,ENQ3,ENQ2,
     3                   PR2,PR1,ENQ1,EPSJ,DUMMY
      EXTERNAL           FCN,FCNJ
      COMMON /DBAND/     NLC,NUC
      COMMON /GEAR/      T,H,HMIN,HMAX,EPS,UROUND,EPSJ,HUSED,
     1                   EL,OLDL0,TOLD,RMAX,RC,CRATE,EPSOLD,HOLD,FN,
     2                   EDN,E,EUP,BND,RH,R1,R,HL0,R0,D,PHL0,PR3,D1,
     3                   ENQ3,ENQ2,PR2,PR1,ENQ1,DUMMY,TQ,
     4                   N,MF,KFLAG,JSTART,NSQ,NQUSED,NSTEP,NFE,NJE,
     5                   NPW,NERROR,NSAVE1,NSAVE2,NEQUIL,NY,
     6                   I,METH,MITER,NQ,L,IDOUB,MFOLD,NOLD,IRET,MEO,
     7                   MIO,IWEVAL,MAXDER,LMAX,IREDO,J,NSTEPJ,J1,J2,
     8                   M,NEWQ,IDUMMY
C                                  FIRST EXECUTABLE STATEMENT
      KFLAG = 0
      TOLD = T
C                                  THIS ROUTINE PERFORMS ONE STEP OF
C                                    THE INTEGRATION OF AN INITIAL
C                                    VALUE PROBLEM FOR A SYSTEM OF
C                                    ORDINARY DIFFERENTIAL EQUATIONS.
      IF (JSTART.GT.0) GO TO 50
      IF (JSTART.NE.0) GO TO 10
C                                  ON THE FIRST CALL, THE ORDER IS SET
C                                    TO 1 AND THE INITIAL YDOT IS
C                                    CALCULATED. RMAX IS THE MAXIMUM
C                                    RATIO BY WHICH H CAN BE INCREASED
C                                    IN A SINGLE STEP. IT IS INITIALLY
C                                    1.E4 TO COMPENSATE FOR THE SMALL
C                                    INITIAL H, BUT THEN IS NORMALLY
C                                    EQUAL TO 10. IF A FAILURE OCCURS
C                                    (IN CORRECTOR CONVERGENCE OR ERROR
C                                    TEST), RMAX IS SET AT 2 FOR THE
C                                    NEXT INCREASE.
      CALL FCN (N,T,Y,SAVE1)
      DO 5 I=1,N
    5 Y(I,2) = H*SAVE1(I)
      METH = MF/10
      MITER = MF-10*METH
      NQ = 1
      L = 2
      IDOUB = 3
      RMAX = 1.D4
      RC = 0.D0
      CRATE = 1.D0
      HOLD = H
      MFOLD = MF
      NSTEP = 0
      NSTEPJ = 0
      NFE = 1
      NJE = 0
      IRET = 3
      GO TO 15
C                                  IF THE CALLER HAS CHANGED METH,
C                                    DGRCS IS CALLED TO SET THE
C                                    COEFFICIENTS OF THE METHOD. IF THE
C                                    CALLER HAS CHANGED N, EPS, OR
C                                    METH, THE CONSTANTS E, EDN, EUP,
C                                    AND BND MUST BE RESET. E IS A
C                                    COMPARISON FOR ERRORS OF THE
C                                    CURRENT ORDER NQ. EUP IS TO TEST
C                                    FOR INCREASING THE ORDER, EDN FOR
C                                    DECREASING THE ORDER. BND IS USED
C                                    TO TEST FOR CONVERGENCE OF THE
C                                    CORRECTOR ITERATES. IF THE CALLER
C                                    HAS CHANGED H, Y MUST BE RESCALED.
C                                    IF H OR METH HAS BEEN CHANGED,
C                                    IDOUB IS RESET TO L + 1 TO PREVENT
C                                    FURTHER CHANGES IN H FOR THAT MANY
C                                    STEPS.
   10 IF (MF.EQ.MFOLD) GO TO 25
      MEO = METH
      MIO = MITER
      METH = MF/10
      MITER = MF-10*METH
      MFOLD = MF
      IF (MITER.NE.MIO) IWEVAL = MITER
      IF (METH.EQ.MEO) GO TO 25
      IDOUB = L+1
      IRET = 1
   15 CALL DGRCS (METH,NQ,EL,TQ,MAXDER)
      LMAX = MAXDER+1
      RC = RC*EL(1)/OLDL0
      OLDL0 = EL(1)
   20 FN = N
      EDN = FN*(TQ(1)*EPS)**2
      E = FN*(TQ(2)*EPS)**2
      EUP = FN*(TQ(3)*EPS)**2
      BND = FN*(TQ(4)*EPS)**2
      EPSOLD = EPS
      NOLD = N
      GO TO (30,35,50), IRET
   25 IF ((EPS.EQ.EPSOLD).AND.(N.EQ.NOLD)) GO TO 30
      IF (N.EQ.NOLD) IWEVAL = MITER
      IRET = 1
      GO TO 20
   30 IF (H.EQ.HOLD) GO TO 50
      RH = H/HOLD
      H = HOLD
      IREDO = 3
      GO TO 40
   35 RH = DMAX1(RH,HMIN/DABS(H))
   40 RH = DMIN1(RH,HMAX/DABS(H),RMAX)
      R1 = 1.D0
      DO 45 J=2,L
         R1 = R1*RH
      DO 45 I=1,N
   45 Y(I,J) = Y(I,J)*R1
      H = H*RH
      RC = RC*RH
      IDOUB = L+1
      IF (IREDO.EQ.0) GO TO 285
C                                  THIS SECTION COMPUTES THE PREDICTED
C                                    VALUES BY EFFECTIVELY MULTIPLYING
C                                    THE Y ARRAY BY THE PASCAL TRIANGLE
C                                    MATRIX. RC IS THE RATIO OF NEW TO
C                                    OLD VALUES OF THE COEFFICIENT
C                                    H*EL(1). WHEN RC DIFFERS FROM 1 BY
C                                    MORE THAN 30 PERCENT, OR THE
C                                    CALLER HAS CHANGED MITER, IWEVAL
C                                    IS SET TO MITER TO FORCE THE
C                                    PARTIALS TO BE UPDATED, IF
C                                    PARTIALS ARE USED. IN ANY CASE,
C                                    THE PARTIALS ARE UPDATED AT LEAST
C                                    EVERY 20-TH STEP.
   50 IF (DABS(RC-1.D0).GT.0.3D0) IWEVAL = MITER
      IF (NSTEP.GE.NSTEPJ+20) IWEVAL = MITER
      T = T+H
      DO 55 J1=1,NQ
      DO 55 J2=J1,NQ
         J = (NQ+J1)-J2
      DO 55 I=1,N
   55 Y(I,J) = Y(I,J)+Y(I,J+1)
C                                  UP TO 3 CORRECTOR ITERATIONS ARE
C                                    TAKEN. A CONVERGENCE TEST IS MADE
C                                    ON THE R.M.S. NORM OF EACH
C                                    CORRECTION, USING BND, WHICH IS
C                                    DEPENDENT ON EPS. THE SUM OF THE
C                                    CORRECTIONS IS ACCUMULATED IN THE
C                                    VECTOR ERROR(I). THE Y ARRAY IS
C                                    NOT ALTERED IN THE CORRECTOR LOOP.
C                                    THE UPDATED Y VECTOR IS STORED
C                                    TEMPORARILY IN SAVE1.
   60 DO 65 I=1,N
   65 ERROR(I) = 0.D0
      M = 0
      CALL FCN (N,T,Y,SAVE2)
      NFE = NFE+1
      IF (IWEVAL.LE.0) GO TO 95
C                                  IF INDICATED, THE MATRIX P = I -
C                                    H*EL(1)*J IS REEVALUATED BEFORE
C                                    STARTING THE CORRECTOR ITERATION.
C                                    IWEVAL IS SET TO 0 AS AN INDICATOR
C                                    THAT THIS HAS BEEN DONE. IF MITER
C                                    = 1 OR 2, P IS COMPUTED AND
C                                    PROCESSED IN PSET. IF MITER = 3,
C                                    THE MATRIX USED IS P = I -
C                                    H*EL(1)*D, WHERE D IS A DIAGONAL
C                                    MATRIX.
      IWEVAL = 0
      RC = 1.D0
      NJE = NJE+1
      NSTEPJ = NSTEP
      GO TO (75,70,80), MITER
   70 NFE = NFE+N
   75 CON = -H*EL(1)
      MITER1 = MITER
      CALL DGRPS (FCN,FCNJ,Y,N0,CON,MITER1,YMAX,SAVE1,SAVE2,PW,EQUIL,
     1 IPIV,IER)
      IF (IER.NE.0) GO TO 155
      GO TO 125
   80 R = EL(1)*.1D0
      DO 85 I=1,N
   85 PW(I) = Y(I,1)+R*(H*SAVE2(I)-Y(I,2))
      CALL FCN (N,T,PW,SAVE1)
      NFE = NFE+1
      HL0 = H*EL(1)
      DO 90 I=1,N
         R0 = H*SAVE2(I)-Y(I,2)
         PW(I) = 1.D0
         D = .1D0*R0-H*(SAVE1(I)-SAVE2(I))
         SAVE1(I) = 0.D0
         IF (DABS(R0).LT.UROUND*YMAX(I)) GO TO 90
         IF (DABS(D).EQ.0.D0) GO TO 155
         PW(I) = .1D0*R0/D
         SAVE1(I) = PW(I)*R0
   90 CONTINUE
      GO TO 135
   95 IF (MITER.NE.0) GO TO (125,125,105), MITER
C
C                                  IN THE CASE OF FUNCTIONAL ITERATION,
C                                    UPDATE Y DIRECTLY FROM THE RESULT
C                                    OF THE LAST FCN CALL.
      D = 0.D0
      DO 100 I=1,N
         R = H*SAVE2(I)-Y(I,2)
         D = D+((R-ERROR(I))/YMAX(I))**2
         SAVE1(I) = Y(I,1)+EL(1)*R
  100 ERROR(I) = R
      GO TO 145
C                                  IN THE CASE OF THE CHORD METHOD,
C                                    COMPUTE THE CORRECTOR ERROR, F SUB
C                                    (M), AND SOLVE THE LINEAR SYSTEM
C                                    WITH THAT AS RIGHT-HAND SIDE AND P
C                                    AS COEFFICIENT MATRIX, USING THE
C                                    LU DECOMPOSITION IF MITER = 1 OR
C                                    2. IF MITER = 3, THE COEFFICIENT
C                                    H*EL(1) IN P IS UPDATED.
  105 PHL0 = HL0
      HL0 = H*EL(1)
      IF (HL0.EQ.PHL0) GO TO 115
      R = HL0/PHL0
      DO 110 I=1,N
         D = 1.D0-R*(1.D0-1.D0/PW(I))
         IF (DABS(D).EQ.0.D0) GO TO 165
  110 PW(I) = 1.D0/D
  115 DO 120 I=1,N
  120 SAVE1(I) = PW(I)*(H*SAVE2(I)-(Y(I,2)+ERROR(I)))
      GO TO 135
  125 DO 130 I=1,N
  130 SAVE1(I) = H*SAVE2(I)-(Y(I,2)+ERROR(I))
      IF (NLC .EQ. -1) GO TO 131
      NWK = (NLC+NUC+1)*N0+1
      CALL LEQT1B(PW,N,NLC,NUC,N0,SAVE1,1,N0,2,PW(NWK),JER)
      GO TO 135
  131 CALL LUELMF (PW,SAVE1,IPIV,N,N0,SAVE1)
  135 D = 0.D0
      DO 140 I=1,N
         ERROR(I) = ERROR(I)+SAVE1(I)
         D = D+(SAVE1(I)/YMAX(I))**2
  140 SAVE1(I) = Y(I,1)+EL(1)*ERROR(I)
C                                  TEST FOR CONVERGENCE. IF M.GT.0, THE
C                                    SQUARE OF THE CONVERGENCE RATE
C                                    CONSTANT IS ESTIMATED AS CRATE,
C                                    AND THIS IS USED IN THE TEST.
  145 IF (M.NE.0) CRATE = DMAX1(.9D0*CRATE,D/D1)
      IF ((D*DMIN1(1.D0,2.D0*CRATE)).LE.BND) GO TO 170
      D1 = D
      M = M+1
      IF (M.EQ.3) GO TO 150
      CALL FCN (N,T,SAVE1,SAVE2)
      GO TO 95
C                                  THE CORRECTOR ITERATION FAILED TO
C                                    CONVERGE IN 3 TRIES. IF PARTIALS
C                                    ARE INVOLVED BUT ARE NOT UP TO
C                                    DATE, THEY ARE REEVALUATED FOR THE
C                                    NEXT TRY. OTHERWISE THE Y ARRAY IS
C                                    RETRACTED TO ITS VALUES BEFORE
C                                    PREDICTION, AND H IS REDUCED, IF
C                                    POSSIBLE. IF NOT, A NO-CONVERGENCE
C                                    EXIT IS TAKEN.
  150 NFE = NFE+2
      IF (IWEVAL.EQ.-1) GO TO 165
  155 T = TOLD
      RMAX = 2.D0
      DO 160 J1=1,NQ
      DO 160 J2=J1,NQ
         J = (NQ+J1)-J2
      DO 160 I=1,N
  160 Y(I,J) = Y(I,J)-Y(I,J+1)
      IF (DABS(H).LE.HMIN*1.00001D0) GO TO 280
      RH = .25D0
      IREDO = 1
      GO TO 35
  165 IWEVAL = MITER
      GO TO 60
C                                  THE CORRECTOR HAS CONVERGED. IWEVAL
C                                    IS SET TO -1 IF PARTIAL
C                                    DERIVATIVES WERE USED, TO SIGNAL
C                                    THAT THEY MAY NEED UPDATING ON
C                                    SUBSEQUENT STEPS. THE ERROR TEST
C                                    IS MADE AND CONTROL PASSES TO
C                                    STATEMENT 190 IF IT FAILS.
  170 IF (MITER.NE.0) IWEVAL = -1
      NFE = NFE+M
      D = 0.D0
      DO 175 I=1,N
  175 D = D+(ERROR(I)/YMAX(I))**2
      IF (D.GT.E) GO TO 190
C                                  AFTER A SUCCESSFUL STEP, UPDATE THE
C                                    Y ARRAY. CONSIDER CHANGING H IF
C                                    IDOUB = 1. OTHERWISE DECREASE
C                                    IDOUB BY 1. IF IDOUB IS THEN 1 AND
C                                    NQ .LT. MAXDER, THEN ERROR IS
C                                    SAVED FOR USE IN A POSSIBLE ORDER
C                                    INCREASE ON THE NEXT STEP. IF A
C                                    CHANGE IN H IS CONSIDERED, AN
C                                    INCREASE OR DECREASE IN ORDER BY
C                                    ONE IS CONSIDERED ALSO. A CHANGE
C                                    IN H IS MADE ONLY IF IT IS BY A
C                                    FACTOR OF AT LEAST 1.1. IF NOT,
C                                    IDOUB IS SET TO 10 TO PREVENT
C                                    TESTING FOR THAT MANY STEPS.
      KFLAG = 0
      IREDO = 0
      NSTEP = NSTEP+1
      HUSED = H
      NQUSED = NQ
      DO 180 J=1,L
      DO 180 I=1,N
  180 Y(I,J) = Y(I,J)+EL(J)*ERROR(I)
      IF (IDOUB.EQ.1) GO TO 200
      IDOUB = IDOUB-1
      IF (IDOUB.GT.1) GO TO 290
      IF (L.EQ.LMAX) GO TO 290
      DO 185 I=1,N
  185 Y(I,LMAX) = ERROR(I)
      GO TO 290
C                                  THE ERROR TEST FAILED. KFLAG KEEPS
C                                    TRACK OF MULTIPLE FAILURES.
C                                    RESTORE T AND THE Y ARRAY TO THEIR
C                                    PREVIOUS VALUES, AND PREPARE TO
C                                    TRY THE STEP AGAIN. COMPUTE THE
C                                    OPTIMUM STEP SIZE FOR THIS OR ONE
C                                    LOWER ORDER.
  190 KFLAG = KFLAG-1
      T = TOLD
      DO 195 J1=1,NQ
      DO 195 J2=J1,NQ
         J = (NQ+J1)-J2
      DO 195 I=1,N
  195 Y(I,J) = Y(I,J)-Y(I,J+1)
      RMAX = 2.D0
      IF (DABS(H).LE.HMIN*1.00001D0) GO TO 270
      IF (KFLAG.LE.-3) GO TO 260
      IREDO = 2
      PR3 = 1.D+20
      GO TO 210
C                                  REGARDLESS OF THE SUCCESS OR FAILURE
C                                    OF THE STEP, FACTORS PR1, PR2, AND
C                                    PR3 ARE COMPUTED, BY WHICH H COULD
C                                    BE DIVIDED AT ORDER NQ - 1, ORDER
C                                    NQ, OR ORDER NQ + 1, RESPECTIVELY.
C                                    IN THE CASE OF FAILURE, PR3 =
C                                    1.E20 TO AVOID AN ORDER INCREASE.
C                                    THE SMALLEST OF THESE IS
C                                    DETERMINED AND THE NEW ORDER
C                                    CHOSEN ACCORDINGLY. IF THE ORDER
C                                    IS TO BE INCREASED, WE COMPUTE ONE
C                                    ADDITIONAL SCALED DERIVATIVE.
  200 PR3 = 1.D+20
      IF (L.EQ.LMAX) GO TO 210
      D1 = 0.D0
      DO 205 I=1,N
  205 D1 = D1+((ERROR(I)-Y(I,LMAX))/YMAX(I))**2
      ENQ3 = .5D0/(L+1)
      PR3 = ((D1/EUP)**ENQ3)*1.4D0+1.4D-6
  210 ENQ2 = .5D0/L
      PR2 = ((D/E)**ENQ2)*1.2D0+1.2D-6
      PR1 = 1.D+20
      IF (NQ.EQ.1) GO TO 220
      D = 0.D0
      DO 215 I=1,N
  215 D = D+(Y(I,L)/YMAX(I))**2
      ENQ1 = .5D0/NQ
      PR1 = ((D/EDN)**ENQ1)*1.3D0+1.3D-6
  220 IF (PR2.LE.PR3) GO TO 225
      IF (PR3.LT.PR1) GO TO 235
      GO TO 230
  225 IF (PR2.GT.PR1) GO TO 230
      NEWQ = NQ
      RH = 1.D0/PR2
      GO TO 250
  230 NEWQ = NQ-1
      RH = 1.D0/PR1
      IF (KFLAG.NE.0.AND.RH.GT.1.D0) RH = 1.D0
      GO TO 250
  235 NEWQ = L
      RH = 1.D0/PR3
      IF (RH.LT.1.1D0) GO TO 245
      DO 240 I=1,N
  240 Y(I,NEWQ+1) = ERROR(I)*EL(L)/L
      GO TO 255
  245 IDOUB = 10
      GO TO 290
  250 IF ((KFLAG.EQ.0).AND.(RH.LT.1.1D0)) GO TO 245
C
C                                  IF THERE IS A CHANGE OF ORDER, RESET
C                                    NQ, L, AND THE COEFFICIENTS. IN
C                                    ANY CASE H IS RESET ACCORDING TO
C                                    RH AND THE Y ARRAY IS RESCALED.
C                                    THEN EXIT FROM 285 IF THE STEP WAS
C                                    OK, OR REDO THE STEP OTHERWISE.
      IF (NEWQ.EQ.NQ) GO TO 35
  255 NQ = NEWQ
      L = NQ+1
      IRET = 2
      GO TO 15
C                                  CONTROL REACHES THIS SECTION IF 3 OR
C                                    MORE FAILURES HAVE OCCURED. IT IS
C                                    ASSUMED THAT THE DERIVATIVES THAT
C                                    HAVE ACCUMULATED IN THE Y ARRAY
C                                    HAVE ERRORS OF THE WRONG ORDER.
C                                    HENCE THE FIRST DERIVATIVE IS
C                                    RECOMPUTED, AND THE ORDER IS SET
C                                    TO 1. THEN H IS REDUCED BY A
C                                    FACTOR OF 10, AND THE STEP IS
C                                    RETRIED. AFTER A TOTAL OF 7
C                                    FAILURES, AN EXIT IS TAKEN WITH
C                                    KFLAG = -2.
  260 IF (KFLAG.EQ.-7) GO TO 275
      RH = .1D0
      RH = DMAX1(HMIN/DABS(H),RH)
      H = H*RH
      CALL FCN (N,T,Y,SAVE1)
      NFE = NFE+1
      DO 265 I=1,N
  265 Y(I,2) = H*SAVE1(I)
      IWEVAL = MITER
      IDOUB = 10
      IF (NQ.EQ.1) GO TO 50
      NQ = 1
      L = 2
      IRET = 3
      GO TO 15
C                                  ALL RETURNS ARE MADE THROUGH THIS
C                                    SECTION. H IS SAVED IN HOLD TO
C                                    ALLOW THE CALLER TO CHANGE H ON
C                                    THE NEXT STEP.
  270 KFLAG = -1
      GO TO 290
  275 KFLAG = -2
      GO TO 290
  280 KFLAG = -3
      GO TO 290
  285 RMAX = 10.D0
  290 HOLD = H
      JSTART = NQ
      RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JANUARY 1, 1978
C
C   PURPOSE             - LINEAR EQUATION SOLUTION - BAND STORAGE
C                           MODE - SPACE ECONOMIZER SOLUTION
C
C   USAGE               - CALL LEQT1B (A,N,NLC,NUC,IA,B,M,IB,IJOB,XL,
C                           IER)
C
C   ARGUMENTS    A      - INPUT/OUTPUT MATRIX OF DIMENSION N BY
C                           (NUC+NLC+1). SEE PARAMETER IJOB.
C                N      - ORDER OF MATRIX A AND THE NUMBER OF ROWS IN
C                           B. (INPUT)
C                NLC    - NUMBER OF LOWER CODIAGONALS IN MATRIX A.
C                           (INPUT)
C                NUC    - NUMBER OF UPPER CODIAGONALS IN MATRIX A.
C                           (INPUT)
C                IA     - ROW DIMENSION OF MATRIX A EXACTLY AS
C                           SPECIFIED IN THE DIMENSION STATEMENT IN THE
C                           CALLING PROGRAM. (INPUT)
C                B      - INPUT/OUTPUT MATRIX OF DIMENSION N BY M.
C                           ON INPUT, B CONTAINS THE M RIGHT-HAND SIDES
C                           OF THE EQUATION AX = B. ON OUTPUT, THE
C                           SOLUTION MATRIX X REPLACES B. IF IJOB = 1,
C                           B IS NOT USED.
C                M      - NUMBER OF RIGHT HAND SIDES (COLUMNS IN B).
C                           (INPUT)
C                IB     - ROW DIMENSION OF MATRIX B EXACTLY AS
C                           SPECIFIED IN THE DIMENSION STATEMENT IN THE
C                           CALLING PROGRAM. (INPUT)
C                IJOB   - INPUT OPTION PARAMETER. IJOB = I IMPLIES WHEN
C                           I = 0, FACTOR THE MATRIX A AND SOLVE THE
C                             EQUATION AX = B. ON INPUT, A CONTAINS THE
C                             COEFFICIENT MATRIX OF THE EQUATION AX = B,
C                             WHERE A IS ASSUMED TO BE AN N BY N BAND
C                             MATRIX. A IS STORED IN BAND STORAGE MODE
C                             AND THEREFORE HAS DIMENSION N BY
C                             (NLC+NUC+1). ON OUTPUT, A IS REPLACED
C                             BY THE U MATRIX OF THE L-U DECOMPOSITION
C                             OF A ROWWISE PERMUTATION OF MATRIX A. U
C                             IS STORED IN BAND STORAGE MODE.
C                           I = 1, FACTOR THE MATRIX A. A CONTAINS THE
C                             SAME INPUT/OUTPUT INFORMATION AS IF
C                             IJOB = 0.
C                           I = 2, SOLVE THE EQUATION AX = B. THIS
C                             OPTION IMPLIES THAT LEQT1B HAS ALREADY
C                             BEEN CALLED USING IJOB = 0 OR 1 SO THAT
C                             THE MATRIX A HAS ALREADY BEEN FACTORED.
C                             IN THIS CASE, OUTPUT MATRICES A AND XL
C                             MUST HAVE BEEN SAVED FOR REUSE IN THE
C                             CALL TO LEQT1B.
C                XL     - WORK AREA OF DIMENSION N*(NLC+1). THE FIRST
C                           NLC*N LOCATIONS OF XL CONTAIN COMPONENTS OF
C                           THE L MATRIX OF THE L-U DECOMPOSITION OF A
C                           ROWWISE PERMUTATION OF A. THE LAST N
C                           LOCATIONS CONTAIN THE PIVOT INDICES.
C                IER    - ERROR PARAMETER. (OUTPUT)
C                         TERMINAL ERROR
C                           IER = 129 INDICATES THAT MATRIX A IS
C                             ALGORITHMICALLY SINGULAR. (SEE THE
C                             CHAPTER L PRELUDE).
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - UERTST,UGETIO
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE LEQT1B (A,N,NLC,NUC,IA,B,M,IB,IJOB,XL,IER)
C
      DIMENSION          A(IA,*),XL(N,*),B(IB,*)
      DOUBLE PRECISION   A,XL,B,P,Q,ZERO,ONE,RN
      DATA               ZERO/0.D0/,ONE/1.0D0/
C                                  FIRST EXECUTABLE STATEMENT
      IER = 0
      JBEG = NLC+1
      NLC1 = JBEG
      IF (IJOB .EQ. 2) GO TO 80
      RN = N
C                                  RESTRUCTURE THE MATRIX
C                                  FIND RECIPROCAL OF THE LARGEST
C                                  ABSOLUTE VALUE IN ROW I
      I = 1
      NC = JBEG+NUC
      NN = NC
      JEND = NC
      IF (N .EQ. 1 .OR. NLC .EQ. 0) GO TO 25
    5 K = 1
      P = ZERO
      DO 10 J = JBEG,JEND
         A(I,K) = A(I,J)
         Q =  DABS(A(I,K))
         IF (Q .GT. P) P = Q
         K = K+1
   10 CONTINUE
      IF (P .EQ. ZERO) GO TO 135
      XL(I,NLC1) = ONE/P
      IF (K .GT. NC) GO TO 20
      DO 15 J = K,NC
         A(I,J) = ZERO
   15 CONTINUE
   20 I = I+1
      JBEG = JBEG-1
      IF (JEND-JBEG .EQ. N) JEND = JEND-1
      IF (I .LE. NLC) GO TO 5
      JBEG = I
      NN = JEND
   25 JEND = N-NUC
      DO 40 I = JBEG,N
         P = ZERO
         DO 30 J = 1,NN
            Q =  DABS(A(I,J))
            IF (Q .GT. P) P = Q
   30    CONTINUE
         IF (P .EQ. ZERO) GO TO 135
         XL(I,NLC1) = ONE/P
         IF (I .EQ. JEND) GO TO 37
         IF (I .LT. JEND) GO TO 40
         K = NN+1
         DO 35 J = K,NC
            A(I,J) = ZERO
   35    CONTINUE
   37    NN = NN-1
   40 CONTINUE
      L = NLC
C                                  L-U DECOMPOSITION
      DO 75 K = 1,N
         P =  DABS(A(K,1))*XL(K,NLC1)
         I = K
         IF (L .LT. N) L = L+1
         K1 = K+1
         IF (K1 .GT. L) GO TO 50
         DO 45 J = K1,L
            Q = DABS(A(J,1))*XL(J,NLC1)
            IF (Q .LE. P) GO TO 45
            P = Q
            I = J
   45    CONTINUE
   50    XL(I,NLC1) = XL(K,NLC1)
         XL(K,NLC1) = I
C                                  SINGULARITY FOUND
         Q = RN+P
         IF (Q .EQ. RN) GO TO 135
C                                  INTERCHANGE ROWS I AND K
         IF (K .EQ. I) GO TO 60
         DO 55 J = 1,NC
            P = A(K,J)
            A(K,J) = A(I,J)
            A(I,J) = P
   55    CONTINUE
   60    IF (K1 .GT. L) GO TO 75
         DO 70 I = K1,L
            P = A(I,1)/A(K,1)
            IK = I-K
            XL(K1,IK) = P
            DO 65 J = 2,NC
               A(I,J-1) = A(I,J)-P*A(K,J)
   65    CONTINUE
         A(I,NC) = ZERO
   70    CONTINUE
   75 CONTINUE
      IF (IJOB .EQ. 1) GO TO 9005
C                                  FORWARD SUBSTITUTION
   80 L = NLC
      DO 105 K = 1,N
         I = idint(XL(K,NLC1))
         IF (I .EQ. K) GO TO 90
         DO 85 J = 1,M
            P = B(K,J)
            B(K,J) = B(I,J)
            B(I,J) = P
   85    CONTINUE
   90    IF (L .LT. N) L = L+1
         K1 = K+1
         IF (K1 .GT. L) GO TO 105
         DO 100 I = K1,L
            IK = I-K
            P = XL(K1,IK)
            DO 95 J = 1,M
               B(I,J) = B(I,J)-P*B(K,J)
   95       CONTINUE
  100    CONTINUE
  105 CONTINUE
C                                  BACKWARD SUBSTITUTION
      JBEG = NUC+NLC
      DO 125 J = 1,M
         L = 1
         K1 = N+1
         DO 120 I = 1,N
            K = K1-I
            P = B(K,J)
            IF (L .EQ. 1) GO TO 115
            DO 110 KK = 2,L
               IK = KK+K
               P = P-A(K,KK)*B(IK-1,J)
  110       CONTINUE
  115       B(K,J) = P/A(K,1)
            IF (L .LE. JBEG) L = L+1
  120    CONTINUE
  125 CONTINUE
      GO TO 9005
  135 IER = 129
 9000 CONTINUE
c      write(*,*) (y(i),i=1,n)
      CALL UERTST(IER,'LEQT1B')
 9005 RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JANUARY 1, 1978
C
C   PURPOSE             - L-U DECOMPOSITION BY THE CROUT ALGORITHM
C                           WITH OPTIONAL ACCURACY TEST.
C
C   USAGE               - CALL LUDATF (A,LU,N,IA,IDGT,D1,D2,IPVT,
C                           EQUIL,WA,IER)
C
C   ARGUMENTS    A      - INPUT MATRIX OF DIMENSION N BY N CONTAINING
C                           THE MATRIX TO BE DECOMPOSED.
C                LU     - REAL OUTPUT MATRIX OF DIMENSION N BY N
C                           CONTAINING THE L-U DECOMPOSITION OF A
C                           ROWWISE PERMUTATION OF THE INPUT MATRIX.
C                           FOR A DESCRIPTION OF THE FORMAT OF LU, SEE
C                           EXAMPLE.
C                N      - INPUT SCALAR CONTAINING THE ORDER OF THE
C                           MATRIX A.
C                IA     - INPUT SCALAR CONTAINING THE ROW DIMENSION OF
C                           MATRICES A AND LU EXACTLY AS SPECIFIED IN
C                           THE CALLING PROGRAM.
C                IDGT   - INPUT OPTION.
C                           IF IDGT IS GREATER THAN ZERO, THE NON-ZERO
C                           ELEMENTS OF A ARE ASSUMED TO BE CORRECT TO
C                           IDGT DECIMAL PLACES.  LUDATF PERFORMS AN
C                           ACCURACY TEST TO DETERMINE IF THE COMPUTED
C                           DECOMPOSITION IS THE EXACT DECOMPOSITION
C                           OF A MATRIX WHICH DIFFERS FROM THE GIVEN
C                           ONE BY LESS THAN ITS UNCERTAINTY.
C                         IF IDGT IS EQUAL TO ZERO, THE ACCURACY TEST
C                           IS BYPASSED.
C                D1     - OUTPUT SCALAR CONTAINING ONE OF THE TWO
C                           COMPONENTS OF THE DETERMINANT. SEE
C                           DESCRIPTION OF PARAMETER D2, BELOW.
C                D2     - OUTPUT SCALAR CONTAINING ONE OF THE
C                           TWO COMPONENTS OF THE DETERMINANT. THE
C                           DETERMINANT MAY BE EVALUATED AS (D1)(2**D2).
C                IPVT   - OUTPUT VECTOR OF LENGTH N CONTAINING THE
C                           PERMUTATION INDICES. SEE DOCUMENT
C                           (ALGORITHM).
C                EQUIL  - OUTPUT VECTOR OF LENGTH N CONTAINING
C                           RECIPROCALS OF THE ABSOLUTE VALUES OF
C                           THE LARGEST (IN ABSOLUTE VALUE) ELEMENT
C                           IN EACH ROW.
C                WA     - ACCURACY TEST PARAMETER, OUTPUT ONLY IF
C                           IDGT IS GREATER THAN ZERO.
C                           SEE ELEMENT DOCUMENTATION FOR DETAILS.
C                IER    - ERROR PARAMETER. (OUTPUT)
C                         TERMINAL ERROR
C                           IER = 129 INDICATES THAT MATRIX A IS
C                             ALGORITHMICALLY SINGULAR. (SEE THE
C                             CHAPTER L PRELUDE).
C                         WARNING ERROR
C                           IER = 34 INDICATES THAT THE ACCURACY TEST
C                             FAILED.  THE COMPUTED SOLUTION MAY BE IN
C                             ERROR BY MORE THAN CAN BE ACCOUNTED FOR
C                             BY THE UNCERTAINTY OF THE DATA.  THIS
C                             WARNING CAN BE PRODUCED ONLY IF IDGT IS
C                             GREATER THAN 0 ON INPUT.  SEE CHAPTER L
C                             PRELUDE FOR FURTHER DISCUSSION.
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - UERTST,UGETIO
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   REMARKS      A TEST FOR SINGULARITY IS MADE AT TWO LEVELS:
C                1.  A ROW OF THE ORIGINAL MATRIX A IS NULL.
C                2.  A COLUMN BECOMES NULL IN THE FACTORIZATION PROCESS.
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE LUDATF (A,LU,N,IA,IDGT,D1,D2,IPVT,EQUIL,WA,IER)
C
      DIMENSION          A(IA,*),LU(IA,*),IPVT(*),EQUIL(*)
      DOUBLE PRECISION   A,LU,D1,D2,EQUIL,WA,ZERO,ONE,FOUR,SIXTN,SIXTH,
     *                   RN,WREL,BIGA,BIG,P,SUM,AI,WI,T,TEST,Q
      DATA               ZERO,ONE,FOUR,SIXTN,SIXTH/0.D0,1.D0,4.D0,
     *                   16.D0,.0625D0/
C                                  FIRST EXECUTABLE STATEMENT
C                                  INITIALIZATION
      IER = 0
      RN = N
      WREL = ZERO
      D1 = ONE
      D2 = ZERO
      BIGA = ZERO
      DO 10 I=1,N
         BIG = ZERO
         DO 5 J=1,N
            P = A(I,J)
            LU(I,J) = P
            P = DABS(P)
            BIG = dmax1(big,P)
    5    CONTINUE
         BIGA = dmax1(biga,BIG)
         IF (BIG .EQ. ZERO) GO TO 110
         EQUIL(I) = ONE/BIG
   10 CONTINUE
      DO 105 J=1,N
         JM1 = J-1
         IF (JM1 .LT. 1) GO TO 40
C                                  COMPUTE U(I,J), I=1,...,J-1
         DO 35 I=1,JM1
            SUM = LU(I,J)
            IM1 = I-1
            IF (IDGT .ne. 0) then
C                                  WITH ACCURACY TEST
              AI = DABS(SUM)
              WI = ZERO
              IF (IM1 .ge. 1) then
                DO 15 K=1,IM1
                  T = LU(I,K)*LU(K,J)
                  SUM = SUM-T
                  WI = WI+DABS(T)
   15           CONTINUE
                LU(I,J) = SUM
              endif
              WI = WI+DABS(SUM)
              IF (AI .EQ. ZERO) AI = BIGA
              TEST = WI/AI
              WREL = dmax1(wrel,TEST)

            else
C                                  WITHOUT ACCURACY
   25         IF (IM1 .ge. 1) then
                DO 30 K=1,IM1
                  SUM = SUM-LU(I,K)*LU(K,J)
   30           CONTINUE
                LU(I,J) = SUM
              endif
           endif
   35    CONTINUE
   40    P = ZERO
C                                  COMPUTE U(J,J) AND L(I,J), I=J+1,...,
         DO 70 I=J,N
            SUM = LU(I,J)
            IF (IDGT .EQ. 0) GO TO 55
C                                  WITH ACCURACY TEST
            AI = DABS(SUM)
            WI = ZERO
            IF (JM1 .LT. 1) GO TO 50
            DO 45 K=1,JM1
               T = LU(I,K)*LU(K,J)
               SUM = SUM-T
               WI = WI+DABS(T)
   45       CONTINUE
            LU(I,J) = SUM
   50       WI = WI+DABS(SUM)
            IF (AI .EQ. ZERO) AI = BIGA
            TEST = WI/AI
            IF (TEST .GT. WREL) WREL = TEST
            GO TO 65
C                                  WITHOUT ACCURACY TEST
   55       IF (JM1 .LT. 1) GO TO 65
            DO 60 K=1,JM1
               SUM = SUM-LU(I,K)*LU(K,J)
   60       CONTINUE
            LU(I,J) = SUM
   65       Q = EQUIL(I)*DABS(SUM)
            IF (P .GE. Q) GO TO 70
            P = Q
            IMAX = I
   70    CONTINUE
C                                  TEST FOR ALGORITHMIC SINGULARITY
         Q = RN+P
   71    IF (Q .EQ. RN) GO TO 110
         IF (J .EQ. IMAX) GO TO 80
C                                  INTERCHANGE ROWS J AND IMAX
         D1 = -D1
         DO 75 K=1,N
            P = LU(IMAX,K)
            LU(IMAX,K) = LU(J,K)
            LU(J,K) = P
   75    CONTINUE
         EQUIL(IMAX) = EQUIL(J)
   80    IPVT(J) = IMAX
         D1 = D1*LU(J,J)
   85    IF (DABS(D1) .LE. ONE) GO TO 90
         D1 = D1*SIXTH
         D2 = D2+FOUR
         GO TO 85
   90    IF (DABS(D1) .GE. SIXTH) GO TO 95
         D1 = D1*SIXTN
         D2 = D2-FOUR
         GO TO 90
   95    CONTINUE
         JP1 = J+1
         IF (JP1 .GT. N) GO TO 105
C                                  DIVIDE BY PIVOT ELEMENT U(J,J)
         P = LU(J,J)
         DO 100 I=JP1,N
            LU(I,J) = LU(I,J)/P
  100    CONTINUE
  105 CONTINUE
C                                  PERFORM ACCURACY TEST
      IF (IDGT .EQ. 0) GO TO 9005
      P = 3*N+3
      WA = P*WREL
      Q = WA+10.D0**(-IDGT)
  106 IF (Q .NE. WA) GO TO 9005
      IER = 34
      GO TO 9000
C                                  ALGORITHMIC SINGULARITY
  110 IER = 129
      D1 = ZERO
      D2 = ZERO
 9000 CONTINUE
C                                  PRINT ERROR
      CALL UERTST(IER,'LUDATF')
 9005 RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/DOUBLE
C
C   LATEST REVISION     - JANUARY 1, 1978
C
C   PURPOSE             - ELIMINATION PART OF SOLUTION OF AX=B
C                           (FULL STORAGE MODE)
C
C   USAGE               - CALL LUELMF (A,B,IPVT,N,IA,X)
C
C   ARGUMENTS    A      - A = LU (THE RESULT COMPUTED IN THE IMSL
C                           ROUTINE LUDATF) WHERE L IS A LOWER
C                           TRIANGULAR MATRIX WITH ONES ON THE MAIN
C                           DIAGONAL. U IS UPPER TRIANGULAR. L AND U
C                           ARE STORED AS A SINGLE MATRIX A AND THE
C                           UNIT DIAGONAL OF L IS NOT STORED. (INPUT)
C                B      - B IS A VECTOR OF LENGTH N ON THE RIGHT HAND
C                           SIDE OF THE EQUATION AX=B. (INPUT)
C                IPVT   - THE PERMUTATION MATRIX RETURNED FROM THE
C                           IMSL ROUTINE LUDATF, STORED AS AN N LENGTH
C                           VECTOR. (INPUT)
C                N      - ORDER OF A AND NUMBER OF ROWS IN B. (INPUT)
C                IA     - ROW DIMENSION OF A EXACTLY AS SPECIFIED IN
C                           THE DIMENSION STATEMENT IN THE CALLING
C                           PROGRAM. (INPUT)
C                X      - THE RESULT X. (OUTPUT)
C
C   PRECISION/HARDWARE  - SINGLE AND DOUBLE/H32
C                       - SINGLE/H36,H48,H60
C
C   REQD. IMSL ROUTINES - NONE REQUIRED
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE LUELMF (A,B,IPVT,N,IA,X)
C
      DIMENSION          A(IA,*),B(*),IPVT(*),X(*)
      DOUBLE PRECISION   A,B,X,SUM
C                                  FIRST EXECUTABLE STATEMENT
C                                  SOLVE LY = B FOR Y
      DO 5 I=1,N
    5 X(I) = B(I)
      IW = 0
      DO 20 I=1,N
         IP = IPVT(I)
         SUM = X(IP)
         X(IP) = X(I)
         IF (IW .EQ. 0) then
           IF (SUM .NE. 0.D0) IW = I
         else
           IM1 = I-1
           DO 10 J=IW,IM1
             SUM = SUM-A(I,J)*X(J)
   10      CONTINUE
         endif
   20 X(I) = SUM
C                                  SOLVE UX = Y FOR X
      DO 30 IB=1,N
         I = N+1-IB
         IP1 = I+1
         SUM = X(I)
         IF (IP1 .le. N) then
           DO 25 J=IP1,N
             SUM = SUM-A(I,J)*X(J)
   25      CONTINUE
         endif
   30 X(I) = SUM/A(I,I)
      RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/SINGLE
C
C   LATEST REVISION     - JUNE 1, 1982
C
C   PURPOSE             - PRINT A MESSAGE REFLECTING AN ERROR CONDITION
C
C   USAGE               - CALL UERTST (IER,NAME)
C
C   ARGUMENTS    IER    - ERROR PARAMETER. (INPUT)
C                           IER = I+J WHERE
C                             I = 128 IMPLIES TERMINAL ERROR MESSAGE,
C                             I =  64 IMPLIES WARNING WITH FIX MESSAGE,
C                             I =  32 IMPLIES WARNING MESSAGE.
C                             J = ERROR CODE RELEVANT TO CALLING
C                                 ROUTINE.
C                NAME   - A CHARACTER STRING OF LENGTH SIX PROVIDING
C                           THE NAME OF THE CALLING ROUTINE. (INPUT)
C
C   PRECISION/HARDWARE  - SINGLE/ALL
C
C   REQD. IMSL ROUTINES - UGETIO,USPKD
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   REMARKS      THE ERROR MESSAGE PRODUCED BY UERTST IS WRITTEN
C                TO THE STANDARD OUTPUT UNIT. THE OUTPUT UNIT
C                NUMBER CAN BE DETERMINED BY CALLING UGETIO AS
C                FOLLOWS..   CALL UGETIO(1,NIN,NOUT).
C                THE OUTPUT UNIT NUMBER CAN BE CHANGED BY CALLING
C                UGETIO AS FOLLOWS..
C                                NIN = 0
C                                NOUT = NEW OUTPUT UNIT NUMBER
C                                CALL UGETIO(3,NIN,NOUT)
C                SEE THE UGETIO DOCUMENT FOR MORE DETAILS.
C
C   COPYRIGHT           - 1982 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE UERTST (IER,NAME)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            IER
      CHARACTER          NAME*(*)
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            I,IEQDF,IOUNIT,LEVEL,LEVOLD,NIN
      CHARACTER          IEQ,NAMEQ(6),NAMSET(6),NAMUPK(6)
      DATA               NAMSET/'U','E','R','S','E','T'/
      DATA               NAMEQ/6*' '/
      DATA               LEVEL/4/,IEQDF/0/,IEQ/'='/
C                                  UNPACK NAME INTO NAMUPK
C                                  FIRST EXECUTABLE STATEMENT
Cws   CALL USPKD (NAME,6,NAMUPK,NMTB)
C                                  GET OUTPUT UNIT NUMBER
      CALL UGETIO(1,NIN,IOUNIT)
C                                  CHECK IER
      IF (IER.GT.999) GO TO 25
      IF (IER.LT.-32) GO TO 55
      IF (IER.LE.128) GO TO 5
      IF (LEVEL.LT.1) GO TO 30
C                                  PRINT TERMINAL MESSAGE
      IF (IEQDF.EQ.1) WRITE(IOUNIT,35) IER,NAMEQ,IEQ,NAMUPK
      IF (IEQDF.EQ.0) WRITE(IOUNIT,35) IER,NAMUPK
      GO TO 30
    5 IF (IER.LE.64) GO TO 10
      IF (LEVEL.LT.2) GO TO 30
C                                  PRINT WARNING WITH FIX MESSAGE
cws      IF (IEQDF.EQ.1) WRITE(IOUNIT,40) IER,NAMEQ,IEQ,NAMUPK
cws      IF (IEQDF.EQ.0) WRITE(IOUNIT,40) IER,NAMUPK
      GO TO 30
   10 IF (IER.LE.32) GO TO 15
C                                  PRINT WARNING MESSAGE
      IF (LEVEL.LT.3) GO TO 30
      IF (IEQDF.EQ.1) WRITE(IOUNIT,45) IER,NAMEQ,IEQ,NAMUPK
      IF (IEQDF.EQ.0) WRITE(IOUNIT,45) IER,NAMUPK
      GO TO 30
   15 CONTINUE
C                                  CHECK FOR UERSET CALL
      DO 20 I=1,6
         IF (NAMUPK(I).NE.NAMSET(I)) GO TO 25
   20 CONTINUE
      LEVOLD = LEVEL
      LEVEL = IER
      IER = LEVOLD
      IF (LEVEL.LT.0) LEVEL = 4
      IF (LEVEL.GT.4) LEVEL = 4
      GO TO 30
   25 CONTINUE
      IF (LEVEL.LT.4) GO TO 30
C                                  PRINT NON-DEFINED MESSAGE
      IF (IEQDF.EQ.1) WRITE(IOUNIT,50) IER,NAMEQ,IEQ,NAMUPK
      IF (IEQDF.EQ.0) WRITE(IOUNIT,50) IER,NAMUPK
 30   IEQDF = 0

      RETURN
   35 FORMAT(19H *** TERMINAL ERROR,10X,7H(IER = ,I3,
     1       20H) FROM IMSL ROUTINE ,6A1,A1,6A1)
   40 FORMAT(27H *** WARNING WITH FIX ERROR,2X,7H(IER = ,I3,
     1       20H) FROM IMSL ROUTINE ,6A1,A1,6A1)
   45 FORMAT(18H *** WARNING ERROR,11X,7H(IER = ,I3,
     1       20H) FROM IMSL ROUTINE ,6A1,A1,6A1)
   50 FORMAT(20H *** UNDEFINED ERROR,9X,7H(IER = ,I5,
     1       20H) FROM IMSL ROUTINE ,6A1,A1,6A1)
C
C                                  SAVE P FOR P = R CASE
C                                    P IS THE PAGE NAMUPK
C                                    R IS THE ROUTINE NAMUPK
   55 IEQDF = 1
      DO 60 I=1,6
   60 NAMEQ(I) = NAMUPK(I)
   65 RETURN
      END
C
C-----------------------------------------------------------------------
C
C   COMPUTER            - HP9000/SINGLE
C
C   LATEST REVISION     - JUNE 1, 1981
C
C   PURPOSE             - TO RETRIEVE CURRENT VALUES AND TO SET NEW
C                           VALUES FOR INPUT AND OUTPUT UNIT
C                           IDENTIFIERS.
C
C   USAGE               - CALL UGETIO(IOPT,NIN,NOUT)
C
C   ARGUMENTS    IOPT   - OPTION PARAMETER. (INPUT)
C                           IF IOPT=1, THE CURRENT INPUT AND OUTPUT
C                           UNIT IDENTIFIER VALUES ARE RETURNED IN NIN
C                           AND NOUT, RESPECTIVELY.
C                           IF IOPT=2, THE INTERNAL VALUE OF NIN IS
C                           RESET FOR SUBSEQUENT USE.
C                           IF IOPT=3, THE INTERNAL VALUE OF NOUT IS
C                           RESET FOR SUBSEQUENT USE.
C                NIN    - INPUT UNIT IDENTIFIER.
C                           OUTPUT IF IOPT=1, INPUT IF IOPT=2.
C                NOUT   - OUTPUT UNIT IDENTIFIER.
C                           OUTPUT IF IOPT=1, INPUT IF IOPT=3.
C
C   PRECISION/HARDWARE  - SINGLE/ALL
C
C   REQD. IMSL ROUTINES - NONE REQUIRED
C
C   NOTATION            - INFORMATION ON SPECIAL NOTATION AND
C                           CONVENTIONS IS AVAILABLE IN THE MANUAL
C                           INTRODUCTION OR THROUGH IMSL ROUTINE UHELP
C
C   REMARKS      EACH IMSL ROUTINE THAT PERFORMS INPUT AND/OR OUTPUT
C                OPERATIONS CALLS UGETIO TO OBTAIN THE CURRENT UNIT
C                IDENTIFIER VALUES. IF UGETIO IS CALLED WITH IOPT=2 OR
C                IOPT=3, NEW UNIT IDENTIFIER VALUES ARE ESTABLISHED.
C                SUBSEQUENT INPUT/OUTPUT IS PERFORMED ON THE NEW UNITS.
C
C   COPYRIGHT           - 1978 BY IMSL, INC. ALL RIGHTS RESERVED.
C
C   WARRANTY            - IMSL WARRANTS ONLY THAT IMSL TESTING HAS BEEN
C                           APPLIED TO THIS CODE. NO OTHER WARRANTY,
C                           EXPRESSED OR IMPLIED, IS APPLICABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE UGETIO(IOPT,NIN,NOUT)
C                                  SPECIFICATIONS FOR ARGUMENTS
      integer*4            IOPT,NIN,NOUT
C                                  SPECIFICATIONS FOR LOCAL VARIABLES
      integer*4            NIND,NOUTD
      DATA               NIND/5/,NOUTD/6/

      if     (iopt.eq.1) then
        nin = nind
        nout = noutd
      elseif (iopt.eq.2) then
        nind = nin
      elseif (iopt.eq.3) then
        noutd = nout
      endif

      return
      end
