      program tbc
c
c     ***************************************************
c     *                   <<< TBC >>>                   *
c     *                                                 *
c     *      Transport, Biochemistry, and Chemistry     *
c     *                   Version 2.01                  *
c     *                   September 1999                *
c     *                                                 *
c     *   (c) Dirk Schaefer, Wolfgang Schaefer (1)      *
c     *                Rene Therrien (2)                *
c     *                                                 *
c     * (1) Interdisciplinary Center                    *
c     *     for Scientific Computing                    *
c     *     University of Heidelberg                    *
c     *     INF 368                                     *
c     *     D-69120 Heidelberg                          *
c     *     Germany                                     *
c     *                                                 *
c     * (2) Departement de Geologie et Genie Geologique *
c     *     Universite Laval, Quebec                    *
c     *     Quebec G1k 7P4 Canada                       *
c     ***************************************************
c
c
c     Time-marching groundwater flow model in three dimensions
c     employing the ORTHOMIN acceleration iterative
c     method for solution of the matrix equations arising from
c     the Galerkin method or from a finite difference discretization.
c     This version uses 3d brick finite elements
c     (with a finite difference switch) or triangular prisms.
c
c     The numbering scheme for the hexahedral brick element faces is:
c
c       face 1: front face  --> nodes 1,2,6,5
c       face 2: right face  --> nodes 2,3,7,6     4---------3
c       face 3: back face   --> nodes 4,3,7,8     |  e.g.   |
c       face 4: left face   --> nodes 1,4,8,5     | face 5  |
c       face 5: bottom face --> nodes 1,2,3,4     |         |
c       face 6: top face    --> nodes 5,6,7,8     1---------2
c 
c     The numbering scheme for the triangular prism element faces is:
c
c       face 1: bottom face --> nodes 1,2,3
c       face 2: top face    --> nodes 4,5,6   
c       face 3: side face   --> nodes 1,2,5,4   
c       face 4: side face   --> nodes 1,3,6,4   
c       face 5: side face   --> nodes 2,3,6,5   
c
c************************************************************************
c
c   some i/o units used:
c
c   required files (always opened)
c   ------------------------------    
c     unit 10:  file 'tbc.fil' contains prefix of problem filename
c
c     all the input files (i.e. files that must exist before the
c     simulation) have the prefix plus a fixed extension as their name
c
c     all output files (i.e. files created during the simulation)
c     have the prefix plus 'o.ext' (where ext is fixed) as their name
c
c     unit 55:  main input datafile ('prefix.in')
c     unit 66:  main output listing ('prefixo.out')
c
c   optional files (opened if specified)
c   ------------------------------------    
c   input:
c     unit 11: input of nodal coordinates ('prefix.nod')
c     unit 12: input of element incidences ('prefix.inc')
c     unit 15: input file containing heads for restarting simulation
c              (binary) ('prefix.hin')
c     unit 16: input file containing concentrations for restarting the
c              simulation (binary) ('prefix.cin')
c     unit 25: input from random field generator (binary)
c              (elemental k's) ('prefix.gen')
cws            or from surfer file (ASCII)
c     unit 26: input from random field generator (binary)
c              (elemental kd's) the filename must be specified in unit 55.
cws            or from surfer file (ASCII)
c
c   output:
c     unit 40: output file containing heads at specified output
c              times (binary) ('prefixo.hds')
c     unit 41: output file containing concentrations at specified output
c              times (binary) ('prefixo.con')
c     unit 42: output of flow mass balance results, used for a
c              transport simulation based on steady-state flow simulation
c              and made with a separate code (binary) ('prefixo.bal')
c     unit 43: output of elemental velocities (binary) ('prefixo.vel')
c
c     unit 50: output of heads for last time
c              (for subsequent restart) (binary) ('prefixo.hen')
c     unit 51: output of concentrations for last time
c              (for subsequent restart) (binary) ('prefixo.cen')
c     unit 60: output of nodal fluxes at observation well
c              (ascii) ('prefixo.obs')
c     unit 62: output of nodal fluxes at specified nodes
c              (ascii) ('prefixo.flu')
c     unit 92: output of flux-averaged concentration at wells
c              (ascii) ('prefixo.wec')
c     unit 63: output of concentrations at observation nodes
c              (ascii) ('prefixo.cob')
c     unit 64: output of concentrations at given times
c              (ascii) (' [time] .dat')
c     unit 65: output of mass balance
c              (ascii) ('prefixo.mba')
c     unit 66: general output
c     unit 90: LOG-file for debugging (ascii)
c
c************************************************************************
c
c     array dimensions 
c     file 'tbc.prm' must exist and specify the dimensions
c
c maxbc2   = Maximum number of NODES comprising 2nd-type
c             boundary faces.
c maxkzn   = Maximum number of different hydraulic conductivity zones.
c            If the random generator is used, maxkzn must be at least equal
c            to the number of elements, otherwise set it equal to maxpzn.
c maxnb    = Maximum number of connections for a node.
c            3d blocks: (must be 27 for f.e., 7 for f.d.)
c            3d prisms: (variable)
c maxndc   = Maximum number of dirichlet nodes.
c maxne    = Maximum number of 3D elements.
c maxnja   = Maximum number of nodal connections for the entire
c            grid. The nodal connections for each node, including
c            itself are determined and added up in the set_iaja routine.
c maxnja2  = Maximum number of nodal connections for the entire
c            grid for the decomposed matrix. Maxnja2 is equal to
c            maxnja when a first-order decomposition is selected.
c            Otherwise, for a second-order decomposition, its value
c            must be set higher.
c maxnln   = Maximum number of nodes per 3D element.
c            (6 for triangular prisms, otherwise 8)
c maxnn    = Maximum number of nodes.
c maxnt    = Maximum number of target time steps.
c maxntcrl = Maximum number of generated time steps.
c maxnx    = Maximum number of nodes in x-direction.
c maxny    = Maximum number of nodes in y-direction.
c maxnz    = Maximum number of nodes in z-direction.
c maxobw   = Maximum number of observation (passive) wells.
c maxobwn  = Maximum number of observation well nodes (Sum for all the wells)
c            It is defined as maxobsw*maxnz where it is assumed that the
c            wells are in z-direction.
c maxpzn   = Maximum number of different property zones (or materials).
c maxsp    = Maximum number of species for which physical/chemical
c            transport will be simulated. The following should be used:
c            1, when only a tracer is considered (normal case).
c            10, when the nitrate system is considered.
c            13, when the nitrate and calcite systems are considered.
c maxspm   = Maximum number of mobile species for which physical/chemical
c            transport will be simulated. The following should be used:
c            1, when only a tracer is considered (normal case).
c            4, when the nitrate system is considered.
c            7, when the nitrate and calcite systems are considered.
c maxwell  = Maximum number of injection/withdrawal wells.
c maxwelle = Maximum number of elements containing well nodes.
c maxwellf = Maximum number of fracture elements containing well nodes.
c maxwelln = Maximum number of well nodes (sum for all the wells).
c            It is automatically set to maxwell*maxnz and it is assumed
c            that the wells are in z-direction.
c north    = number of ORTHOMIN orthogonalizations before restart (use 4).
c
c Transport simulation
c --------------------
c   Note that if only flow is simulated, the following parameters can
c   all be set equal to 1 to reduce the amount of storage required.
c 
c maxcobc1 = Maximum number of different concentration values prescribed
c             in time for a specific first-type zone.
c maxcobc3 = Maximum number of different concentration values prescribed
c             in time for a specific third-type zone.
c maxdisn  = Maximum number of nodes where a there is dissolution
c            of an immiscible phase.
c maxkdzn  = Maximum number of different kd zones.
c            if the random generator is used, maxkdzn must be at least equal
c            to the number of elements, otherwise set it equal to maxpznc.
c maxnbc1  = Maximum number of first-type nodes for transport.
c maxnbc3  = Maximum number of elements having faces on 3rd-type
c            boundary condition for transport.
c maxnec   = Maximum number of 3D elements.
c maxnnc   = Maximum number of nodes.
c maxpznc  = Maximum number of different property zones (or materials).
c maxznbc1 = Maximum number of different first-type zones for transport.
c maxznbc3 = Maximum number of different third-type zones for transport.
c
c common blocks
c -------------
c
c The file 'tbc.dim' contains all the common blocks
c and should be present upon compilation
c
c************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      character*100 prefix,name,fname
      integer*4 kout1(maxsp), kout2(maxcspec)

      common /chemanz/ ncspec,nckomp,ncequa,nkspec
      integer*4        ncspec,nckomp,ncequa,nkspec

      common /chemextern/ rconst0,dh0,t0,chemc,compcomp,bspeccomp,
     &                    conversion,
     &                    areac,disspec,compchem,cname,kname
      double precision rconst0(maxcequa),dh0(maxcequa),t0(maxcequa),
     &                 chemc(maxnn*maxcspec),compcomp(maxckomp,maxsp),
     &                 bspeccomp(maxsp,maxcspec)
      integer*4 areac(maxcequa,maxcspec),disspec(maxcequa)
      character*20  cname(maxcspec),kname(maxckomp)
      logical compchem

      common /kinetchem/ rckin0,rckin,stoech,cxalt,dt,reack,
     &                   kinetspec,kkomp,idepend
      double precision rckin0(maxkspec,maxkreac),
     &                  rckin(maxkspec,maxkreac),dt,
     &     stoech(maxkspec,maxckomp),cxalt(maxkspec),
     &     reack(maxkspec,maxcspec,maxkreac)
      integer kinetspec(maxkspec),
     &        kkomp(maxkspec),idepend(maxkspec)

c
c  ...Declare some internal arrays
c     
      dimension jlump(8,4),jlump2(6,4)
      dimension iconxyb(8,3),iconyzb(8,3),iconxzb(8,3)

      data jlump /1,1,2,1,1,2,3,4, 2,2,3,3,5,5,6,5, 4,3,4,4,6,6,7,7,
     &            5,6,7,8,8,7,8,8/
      data jlump2 /1,1,1,1,2,3, 2,2,2,4,4,4, 3,3,3,5,5,5,
     &             4,5,6,6,6,6/
      data iconxyb /3,4,1,2,1,2,1,2, 5,6,5,6,3,4,3,4,
     &              7,8,7,8,7,8,5,6/
      data iconyzb /2,1,4,3,3,3,1,1, 7,7,5,5,4,4,2,2,
     &              8,8,6,6,6,5,8,7/
      data iconxzb /4,3,2,1,2,1,1,2, 6,5,5,6,3,4,4,3,
     &              7,8,8,7,8,7,6,5/

c      double precision cputime

c************************************************************************

c      cpu1=dble(mclock())/100.d0

      half=0.5d0
      third=1.0d0/3.0d0
      fourth=0.25d0
      sixth=1.0d0/6.0d0
      eight=1.0d0/8.0d0
      xninth=1.0d0/9.0d0

      dsmall = 1.0D-6

      write(*,*) ' '
      write(*,*) '         ********************************************'
      write(*,*) '         *                                          *'
      write(*,*) '         *                    TBC                   *'
      write(*,*) '         *  TRANSPORT, BIOCHEMISTRY and CHEMISTRY   *'
      write(*,*) '         *                                          *'
      write(*,*) '         *               Dirk Schaefer              *'
      write(*,*) '         *             Wolfgang Schaefer            *'
      write(*,*) '         *               Rene Therrien              *'
      write(*,*) '         *                                          *'
      write(*,*) '         *             Version 2.01 1999            *'
      write(*,*) '         *                                          *'
      write(*,*) '         ********************************************'
      write(*,*) ' '
c
c  ...Open required units
c
   20 format(a60)
c

      open(10,file='tbc.fil',status='old')
      read(10,20) prefix
      lenprefix=index(prefix,' ') - 1
      read(10,20) name
      lenname=index(name,' ') - 1
      close(10)

      write(*,'(2a,/)') ' Reading Input Data from ',
     &                     prefix(:lenprefix)//name(:lenname)//'.in'


c  ...Unit 90: log-file for debugging
c      open(90,file=prefix(:lenprefix)//name(:lenname)//'o.log',
c     &         status='unknown')
c      rewind(90)


c  ...Unit 55: general input file
      open(55,file=prefix(:lenprefix)//name(:lenname)//'.in',
     &         status='old')
      rewind(55)
c  ...Unit 66: general output file
      open(66,file=prefix(:lenprefix)//name(:lenname)//'o.out',
     &         status='unknown')
      rewind(66)
c  ...Unit 65: chemical mass balance
      open(65,file=prefix(:lenprefix)//name(:lenname)//'o.mba',
     &         status='unknown')
      rewind(65)
c
c  ...Read input data
c
      call rinput1 (lenprefix,prefix,lenname,name)
      call rinput2 (lenprefix,prefix,lenname,name)
c
c  ...If transport simulation also performed, read transport data
c
      if(transport) then
c        lchem=0
        call input_transport(prefix,lenprefix,name,lenname)
      end if

      if(.not.transport) outfc=.false.
      if(.not.transport) kwrithc=0
c
c  ...Print grid information
c
      call pgrid
c
c     ... output data to observationpoint-concentrations (.cob)

      koutc1=0
      koutc2=0

             do 9990 k=1,nsp
                if (ioutspec(k).ne.0) then
                   koutc1=koutc1+1
                   kout1(koutc1)=k
                endif
 9990        continue

             do 9995 k=nsp+1,ncspec+nsp
                if (ioutspec(k).ne.0) then
                   koutc2=koutc2+1
                   kout2(koutc2)=k-nsp
                endif
 9995        continue

c
      if (obs_wells)
     &   write(63,'(99(a15,1x))') ' Observationnr.',
     &                            ' Time          ',
     &                          (spname(kout1(i)),i=1,koutc1),
     &                          (cname(kout2(i)),i=1,koutc2)


      koutc1=0
             do 9980 k=1,nsp
                if (ioutspec(k).ne.0.and.iabs(phase(k)).eq.1) then
                   koutc1=koutc1+1
                   kout1(koutc1)=k
                endif
 9980        continue

             if(wells) 
     &   write(92,'(99(a15,1x))') ' Wellnr.',
     &                            ' Time          ',
     &                          (spname(kout1(i)),i=1,koutc1)

c
c  ...Reset some flags for output
c
      kpvel2=kpvel
c
c  ...Check size of arrays
c
      call check_size
      if(transport) call check_size_trans
c
c  ...Initialize array jloop indicating the nodal connections
c     within an element (used for assembly)
c   
      do 4260 i=1,8
        do 4256 j=1,8
          jloop(i,j)=0  
 4256   continue
 4260 continue
      do 4270 i=1,nln
        if(finite_diff) then
          nlnj=4
          do 4267 j=1,nlnj
            if(nln.eq.8) jloop(i,j)=jlump(i,j)
            if(nln.eq.6) jloop(i,j)=jlump2(i,j)
 4267     continue
        elseif(.not.finite_diff) then
          nlnj=nln
          do 4269 j=1,nlnj
            jloop(i,j)=j
 4269     continue
        end if
 4270 continue
      do 4280 i=1,8
        do 4278 j=1,3
          iconxy(i,j)=iconxyb(i,j)
          iconxz(i,j)=iconxzb(i,j)
          iconyz(i,j)=iconyzb(i,j)
 4278   continue
 4280 continue
c
c  ...Determine the nodal connectivity
c
      call setup_iaja
c
c  ...Initialise the local influence coefficient matrices
c
      if(nln.eq.8) then
        if(kfdm.eq.0) then
          call coeffe
          if(transport) call coeffe_trans 
        elseif(kfdm.eq.1) then
          call coeffd
          if(transport) call coeffd_trans 
        end if
      end if
c
c  ...Initialise heads for iterative solver
c    
      if(ndc.gt.0) then
        do 285 i=1,ndc
          ci(jm(i))=hbc1(i)
  285   continue
      end if
      do 800 i=1,nn
        cu(i)=ci(i)
  800 continue
c
c  ...Initialise concentrations if transport is also simulated
c
      if(transport) then
        do 850 i=1,nn
          ctemp(i)=cic(i)
  850   continue
      end if
c
c  ...Check if the simulation is performed
c 
      if(.not.solve) then
        write(66,6670)
        write(*,6670)
        stop
      end if
 6670 format(//,' *** Simulation not performed ***',/,
     &          '         Program stopped')
c
c  ... calculate chemical equilibrium with given starting values
c
      if (compchem) then
        write(*,*)
        write(*,*) 'Calculating chemical equilibrium'
        do 8745 i=1,nn*nsp 
           cu(i)   =cu(i)*conversion 
           cic(i)=cic(i)*conversion 
 8745   continue 
        dt=0.d0 
        call chemistry (nn,cic)
        do 3745 i=1,nn*nsp 
           cu(i)   =cu(i)/conversion 
           cic(i)=cic(i)/conversion 
 3745   continue 
      endif 
c
c  ... output of starting concentrations
c
        call fltoas(0,fname,lenfname)
        open(unit=64,file=prefix(:lenprefix)//fname(:lenfname),
     &      status='unknown')
        write(64,'(99(a15,1x))') ' Node          ',
     &                           ' X             ',
     &                           ' Y             ',
     &       ' Z             ',
     &       ' Hydraulic_head ',      
     &                           (spname(isp),isp=1,nsp),
     &                           (cname(isp),isp=1,ncspec),
     &                           ' Ionic_Strength'


        write(64,*) ' Time: ',0.

        do 1486 i=1,nn
          write(64,1487) i,x(i),y(i),z(i),ci(i),
     &               (cic((isp-1)*nn+i),isp=1,nsp),
     &               (chemc((isp-1)*nn+i),isp=1,ncspec+1)
1486    continue
1487    format (i8,1x,98(E15.5,1x))
        close(64)


c
      write(*,'(/,a)') ' Starting Flow Simulation'
      
c
c  ...Set some pointers for the simulation time bookkeeping
c
c      call calcnodev

      if(time_step_control) ntloop=1
cc      last_time=.false.
      itarget_count=1
      delta_keep=delta
      target_reached=.false.
c
c  ...Start the time loop
c
 7900 format(/7x,54('*')/,7x,'* Time:',d12.5,3x,
     &          'Current time step:',d12.5,' *',/,7x,54('*'))
 7912 format(/28('*'),/,'Steady-state flow simulation',/,28('*'))

      do 8000 ntloop_target=1,nts
 8888   continue
c
c  ...Check various pointers
c
        if(time_step_control) then
          time=t(ntloop)
          if(ntloop.eq.1) then
            time_check=time+1.0d-20
            if(time_check.gt.target_time(1)) then
              t(ntloop)=target_time(1)
              time=t(ntloop)
              itarget_count=itarget_count+1
              target_reached=.true.
            end if
          end if
        else 
          ntloop=ntloop_target
          t(ntloop)=target_time(ntloop)
          time=t(ntloop)
        end if
        flow_solution=.true.
        if(steady_state) then
          if(ntloop.gt.1) then
            flow_solution=.false.
            if(.not.transport) goto 9999
          end if
        else
          flow_solution=.true.
        end if
c 
c  ...Solve for flow if transient or if first-time step
c
        if(flow_solution) then
          time=t(ntloop)
          if(.not.steady_state) then
            write(66,7900) t(ntloop),delta
          else
            write(66,7912) 
          end if
c
c  ...Reassign head values from the previous solution
c
          if(ntloop.gt.1) then
            do 8965 i=1,nn
              ci(i)=cu(i)
 8965       continue
          end if
c
c  ...Solve normally if no wells are present
c
          call solve_sat(ntloop)
c
c  ...Start loop for iterating well discharge and drawdowm
c
          if(iter_flux.and.nwell.gt.0) call wellits(ntloop)
c
c  ...Mass balance computation
c
          if(mass_balance) then
            call masbal(ntloop,time)
            write(42) nntot_well
            if(nntot_well.gt.0) then
              write(42) ((jq(i,j),j=1,nn_well(i)),i=1,nwell)
              write(42) (gbwellmb(i),i=1,nntot_well)
            end if
          end if
c
c  ...Reassign values for next time step
c
          if(nwell.gt.0.and.iter_flux) then
            do 5678 i=1,nwell
              hw2(i)=ci(jq(i,1))
              qtot2(i)=qtot(i)
              hw1(i)=hw(i)
 5678       continue
          end if
c
c  ...Observation well output
c
          if(nobsw.gt.0) then
            write(60,5525) nobsw,t(ntloop)
 5525       format(i5,2x,d15.8,',Total # of obs wells, time')
c
c  ...Output nodal heads at observation wells
c     Compute and output nodal fluxes at observation wells
c
            icount=0
            do 5587 i=1,nobsw
              write(60,5526) nn_obsw(i),i
 5526         format(i5,' nnodes for obswell nr',i3,
     &               ' Node,z-elev,hyd. head,flux')
              do 5583 j=1,nn_obsw(i)
                n1=jobs(i,j)
                zb=z(n1)
                sum=0.0d0
                icount=icount+1
                istart=ia(n1)
                iend=ia(n1+1) - 1
                ik=0
                do 5579 k=istart,iend
                  ik=ik+1
                  sum=sum +rowobs(icount,ik)*cu(ja(k))
 5579           continue
                flux=sum-gbobs(icount)
                write(60,5537) n1,zb,cu(n1),flux
 5537           format(i6,4(1x,d15.8))
 5583         continue
 5587       continue
          end if
c
c  ...Compute elemental Darcy fluxes
c
          if(kpvel.gt.0) then
            itemp=ntloop/kpvel
            temp2=dble(itemp)
            temp=dble(ntloop)/dble(kpvel)
            check=temp-temp2
          else
            check=dsmall*1.0d5
          end if
          if(target_reached .or. (check.lt.dsmall)) then
            if(kpvel.gt.0) then
              kprint_flag=1
              if(kpvel2.eq.0) kprint_flag=0
              call compute_flux(kprint_flag)
            end if
          else
            if(transport) then
              kprint_flag=0
              call compute_flux(kprint_flag)
            end if
          end if
c
c  ...Control of output
          if(kphead.gt.0) then
            itemp=ntloop/kphead
            temp2=dble(itemp)
            temp=dble(ntloop)/dble(kphead)
            check=temp-temp2
          else
            check=1.0d5*dsmall
          end if
          if(target_reached.or.(check.lt.dsmall)) then
            if(kphead.gt.0) then
              write(40) t(ntloop)
              if(echo_to_output) then
                write(66,*) ' '
                if(.not.steady_state) then
                  write(66,*) 'Nodal hydraulic head values, time: ',
     &                        t(ntloop)
                else
                  write(66,*) 'Nodal hydraulic head values'
                end if
              end if
cws              write(40) (cu(i),i=1,nn)
              write(40) (cu(i),i=1,nn)
              if(echo_to_output) then
                write(66,6480) (i,cu(i),i=1,nn)
              end if
            end if
          endif
 6480     format(4(i7,1x,d12.5))
c
c  ...Compute the maximum change in pressure over the time step
c
          if(time_step_control.and.control_head) then
            dhead_max=-1.0d30
            do 7560 i=1,nn
              check=dabs(cu(i)-ci(i))
              if(check.gt.dhead_max) dhead_max=check
 7560       continue
          end if

          write(*,'(/,a)') ' Starting Transport Simulation'
c
c  ...End of the flow simulation
c
        end if
c
c  ...If transport simulation is required, solve for concentration
c  ...Store flow result in ctemp and use vector cu for transport
c
        if(transport) then
          flow_solution=.false.
          do 7500 i=1,nn
            cc=ctemp(i)
            ctemp(i)=cu(i)
            cu(i)=cc
 7500     continue
          call driver_trans(ntloop,prefix,lenprefix)
          do 7520 i=1,nn
            cc=ctemp(i)
            ctemp(i)=cu(i)
            cu(i)=cc
 7520     continue
        end if
c
c  ...End time loop
c
c  ...Set new time step if time step control
c
        if(time_step_control) then
          if(itarget_count.gt.nts) goto 9999
          aminvalue=1.0d30
          if(control_head) then
            checkhead=dhead_allowed/dhead_max
            if(checkhead.lt.aminvalue) aminvalue=checkhead
crt
            write(66,*) 'Checkhead:',checkhead
          end if
          if(control_conc) then
            checkconc=dconc_allowed/dconc_max
            if(checkconc.lt.aminvalue) aminvalue=checkconc
crt
            write(66,*) 'Checkconc:',checkconc
          end if
ccc keep old computed delta          delta_keep=delta_keep*aminvalue
          delta_keep=delta*aminvalue
          temp_time=t(ntloop) + delta_keep
crt
          write(66,*) 'New delta:',delta_keep,'  new time:',temp_time
c  ...Check target times
          if(temp_time.gt.target_time(itarget_count)) then
            temp_time=target_time(itarget_count)
            itarget_count=itarget_count+1
cc            if(itarget_count.gt.nts) last_time=.true.
            delta=temp_time-t(ntloop)
            ntloop=ntloop+1
            t(ntloop)=temp_time
            target_reached=.true.
          else
            delta=delta_keep
            ntloop=ntloop+1
            t(ntloop)=temp_time
            target_reached=.false.
          end if
          goto 8888
        else
          if(ntloop.lt.nts) then
            t(ntloop+1)=target_time(ntloop+1)
            delta=t(ntloop+1)-t(ntloop)
          end if
        end if
 8000 continue

 9999 continue
cc        if( (last_time.and.(kwrith.eq.1)) .or.
cc     &      (steady_state.and.(kwrith.eq.1)) ) then
crt
      if(kwrith.eq.1) then
        write(50) t(ntloop)
        write(50) (cu(i),i=1,nn)
      end if
      if(kwrithc.eq.1) then
          call plastcon(ntloop)
      end if
      write (66,5022)
      write (*,5022)
 5022 format (///20x,'**** Normal exit ****'//)

      close(55)
      close(66)
c      cpu=dble(mclock())/100.d0 -cpu1
c      write (*,*) 'cpu-time : ',cputime
      stop
      end

c***************************************************************************

      subroutine rinput1 (lenprefix,prefix,lenname,name)

c***************************************************************************
c
c  ...Read the data for setting up the problem
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      real*4 var1,var2,ak1,ak2,ak3
      character*80 title,grtitle
      character*60 prefix,name,surffile
      logical ex

      integer*4 faceid(6,4),faceid2(6,4),lenprefix,lenname
      data faceid /1,2,4,1,1,5, 2,3,3,4,2,6, 6,7,7,8,3,7,
     &             5,6,8,5,4,8/
      data faceid2 /1,4,1,1,2,0, 2,5,2,3,3,0, 3,6,5,6,6,0,
     &              0,0,4,4,5,0/

   10 format(a80)

c  ...Start reading each data group
c     Note: begin each group with an identifying line
c           the program just reads the line with a dummy variable

c************************************************************************
c   Group 1: Title
c************************************************************************
      write(*,*) ' Reading GROUP 1'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 1'
         stop
      endif
      read(55,10) title
      write(66,*) ' '
      write(66,*) '       ********************************************'
      write(66,*) '       *                                          *'
      write(66,*) '       *                    TBC                   *'
      write(66,*) '       *  TRANSPORT, BIOCHEMISTRY and CHEMISTRY   *'
      write(66,*) '       *                                          *'
      write(66,*) '       *               Dirk Schaefer              *'
      write(66,*) '       *             Wolfgang Schaefer            *'
      write(66,*) '       *               Rene Therrien              *'
      write(66,*) '       *                                          *'
      write(66,*) '       *             Version 2.01 1998            *'
      write(66,*) '       *                                          *'
      write(66,*) '       ********************************************'
      write(66,12) title
   12 format (///,80('*'),//,a80//,80('*'),/)
 
c************************************************************************
c   Group 2: Simulation control parameters
c************************************************************************
 
c
c switches
c --------
c gen_grid:     true, automatically generate the grid
c               false, no generation (grid read)
c finite_diff:  true, finite difference representation
c               false, default finite element
c mass_balance: true, fluid mass balance for every time step
c               flase, no fluid mass balance
c time_step_control: true, variable time stepping based on changes
c                    in the primary variables is used during simulation
c                    false, time steps are determined prior to simulation
c                    Applies for flow and transport (if simulated).
c transport:    true, if simultaneous transport simulation
c               false, if only flow simulation
c solve:        true, solve the equation(s) (normal setting)
c               false, don't solve neither for flow or transport,
c               this is mainly to check all the input before running
c               (especiallly checking the grid for random generation)
c nx:     number of nodes in x-direction
c ny:     number of nodes in y-direction
c nz:     number of nodes in z-direction
c nln:    number of local nodes per element (default=8)
c
      write(*,*) ' Reading GROUP 2'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 2'
         stop
      endif
      read(55,*) finite_diff
c      read(55,*) mass_balance
      mass_balance=.true.
      read(55,*) time_step_control
      read(55,*) transport
      read(55,*) solve

      read(55,*) gen_grid
      read(55,*) nx
      if (nx.gt.maxnx) then
         write(*,*) 'Demanded number of X-nodes (NX)',nx
     $        ,' larger than available (MAXNX) ',maxnx
     $        ,' Increase MAXNX and recompile'
         stop
      endif
      read(55,*) ny
      if (ny.gt.maxny) then
         write(*,*) 'Demanded number of Y-nodes (NY)',ny
     $        ,' larger than available (MAXNY) ',maxny
     $        ,' Increase MAXNY and recompile'
         stop
      endif
      read(55,*) nz
      if (nz.gt.maxnz) then
         write(*,*) 'Demanded number of Z-nodes (NZ)',nz
     $        ,' larger than available (MAXNZ) ',maxnz
     $        ,' Increase MAXNZ and recompile'
         stop
      endif
      read(55,*) nln
c
c  ...If mass balance is performed, open a file to write mass balance
c     results necessary to compute the mass balance for a related
c     transport simulation
c
      if(mass_balance) then
        open(42,file=prefix(:lenprefix)//name(:lenname)//'o.bal',
     &            status='unknown',
     &   form='unformatted')
        rewind(42)
      end if

      if(nln.eq.0) nln=8
      nln2=nln/2
      dnlninv=1.0d0/dble(nln)
      if(gen_grid)then
         nn=nx*ny*nz
         if (nn.gt.maxnn) then
            write(*,*) 'Demanded number of nodes (NN=NX*NY*NZ)',nn
     $        ,' larger than available (MAXNN) ',maxnn
     $        ,' Increase MAXNN and recompile'
            stop
         endif

        if(nln.eq.8) ne=(nx-1)*(ny-1)*(nz-1)
        if(nln.eq.6) ne=2*(nx-1)*(ny-1)*(nz-1)
        if (ne.gt.maxne) then
           write(*,*) 'Demanded number of elements',ne
     $          ,' larger than available (MAXNE) ',maxne
     $          ,' Increase MAXNE and recompile'
           stop
        endif
        nesl=ne/(nz-1)
        nndsl=nn/nz
      endif
      write(66,13) nx,ny,nz,nln
   13 format (//29('*')/,'Simulation Control Parameters'/,29('*')//
     &,'Number of nodes in x-direction (nx):   ',i6/
     &,'Number of nodes in y-direction (ny):   ',i6/
     &,'Number of nodes in z-direction (nz):   ',i6/
     &,'Number of nodes per element (nln):     ',i6/)

      if(gen_grid) write(66,7224)
      if(.not.gen_grid) write(66,7226)

 7224 format('The grid is generated')
 7226 format('No grid generation')

      if(finite_diff) write(66,7228)
      if(.not.finite_diff) write(66,7230)
 7228 format('Finite differences are used')
 7230 format('Finite elements are used')

c      if(mass_balance) write(66,7280)
c      if(.not.mass_balance) write(66,7282)
c 7280 format('Fluid mass balance is computed')
c 7282 format('Fluid mass balance is not computed')

      if(time_step_control) write(66,3388)
      if(.not.time_step_control) write(66,3390)
 3388 format('Automatic control of time step for flow and transport',
     & ' (if simulated)',/,'  Target ',
     &   'time values are specified')
 3390 format('No control of time step for flow and transport')

      if(transport) write(66,3393)
      if(.not.transport) write(66,3394)
 3393 format('Flow and transport are simulated')
 3394 format('Only flow is simulated')

      if(solve) write(66,3397)
      if(.not.solve) write(66,3398)
 3397 format('Simulation is performed')
 3398 format('Simulation is not performed')

      if(gen_grid) kgrid=1
      if(.not.gen_grid) kgrid=0

      if(finite_diff) then
        kfdm=1
        klump=1
      elseif(.not.finite_diff) then
        kfdm=0
        klump=0
      end if
c
c  ...Data to identify nodes on each 2d face
c
      if(nln.eq.8) then
        do 450 i=1,6
          do 440 j=1,4
            iface(i,j,1)=faceid(i,j)
  440     continue
  450   continue
      elseif(nln.ne.8) then
        do 470 i=1,6
          do 460 j=1,4
            iface(i,j,1)=faceid2(i,j)
  460     continue
  470   continue
      end if
 
c************************************************************************
c   Group 3: Output control parameters
c************************************************************************
 
c     kpmsh:   print finite element mesh
c              =2 print nodal coordinates + element incidences
c              =1 print only nodal coordinates
c              =0 do not print mesh information
c
c     kphead:  print times for nodal heads to unit 40 (binary)
c              =n, printout for every n-th time value
c
c     *** for the kphead and kpconc flags, if time_step_control is used,
c     *** there will be output at the target time steps only if their value
c     *** is greater then 0.
c     
c     kpvel:   compute and output velocities to unit 43 (binary)
c              =n, compute every n-th time value
c
c     krestar: read heads saved from last simulation and use as initial
c              conditions
c              =0, don't read heads
c              =1, read the heads from the prefix.hin file (unit 15, binary)
c
c     kwrith:  write heads from the last time step for further
c              use as initial conditions for a new simulation
c              =0, don't write the heads
c              =1, write the heads in the prefixo.hen file (unit 50, binary)
c
c     kpmasb:  output mass balance calculation (if it is computed)
c              =n, output every n'th time value
c              If time step control, the following applies:
c              > 0, output at target times
c              = 0, no output
c              < 0, output at ALL times
c
c     echo_to_output: logical flag
c              true, echo values in output file according to flags above
c              false, no echo in output file
c        if the grid is large, setting echo_to_output true might cause
c        the ascii output file to become too large to handle.
c
      write(*,*) ' Reading GROUP 3'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 3'
         stop
      endif
      read(55,*) kpmsh,kphead,kpvel,krestar,
     &           kwrith,kpmasb
      read(55,*) echo_to_output 
      write(66,1041) kpmsh,kphead,kpvel,krestar,kwrith,kpmasb
 1041 format(//25('*')/,'Output Control Parameters',/,25('*'),//,
     & 'kpmsh:',t20,i5,/,'kphead:',t20,i5,/,
     & 'kpvel:',t20,i5/,
     & 'krestar:',t20,i5,/,'kwrith:',t20,i5,/,'kpmasb:',t20,i5,/)
      if(echo_to_output) then
       write(66,*) 'The above switches cause output to binary ',
     & 'and general (ascii) files '
      else
       write(66,*) 'The above switches cause output to binary file only'
      end if
      write(66,*) ' '

c
c  ...open files
c
      if(kphead.gt.0) then
        open(40,file=prefix(:lenprefix)//name(:lenname)//'o.hds',
     &            status='unknown',form='unformatted')

        rewind(40)
      end if

      if(.not.transport) kpconc=0
      if(kpconc.gt.0) then
        open(41,file=prefix(:lenprefix)//name(:lenname)//'o.con',
     &            status='unknown',form='unformatted')
        rewind(41)
c        open(81,file=prefix(:lenprefix)//name(:lenname)//'a.con',
c     &       status='unknown',form='formatted')
c        rewind(81)
      end if
      if(kpvel.gt.0) then
        open(43,file=prefix(:lenprefix)//name(:lenname)//'o.vel',
     &       status='unknown',form='unformatted')
        rewind(43)
      end if
      if(krestar.eq.1) then
        inquire(file=prefix(:lenprefix)//name(:lenname)//'.hin',
     &           exist=ex)
        if(.not.ex) then
         write(*,*) ' Restart file (.hin) for heads is missing'
         write(66,*) '** Restart file (.hin) for heads is missing **'
         stop
        end if
        open(unit=15,file=prefix(:lenprefix)//name(:lenname)//'.hin',
     &       status='unknown',form='unformatted')
        rewind(15)
      end if
      if(kwrith.eq.1) then
        inquire(file=prefix(:lenprefix)//name(:lenname)//'o.hen',
     &           exist=ex)
        if(ex) then
         write(*,*) ' File for last time values (.hen) will',
     &    ' be overwritten'
         write(66,*) '** File for last time values (.hen) will',
     &    ' be overwritten **'
         write(66,*) ' '
        end if
        open(unit=50,file=prefix(:lenprefix)//name(:lenname)//'o.hen',
     &       status='unknown',form='unformatted')
        rewind(50)
      end if
c
c  ...if time step control, adjust the print flags to only print
c     at target times, if they are greater than 0.
c     (this is because output is wanted at target times).
c
c     (also open for velocity)
c     for velocity, if the flag has been specified greater
c     than 0 and if there is time_step_control, then only output
c     at target times (i.e. set the flag to large number)
c
      if(time_step_control) then
        if(kphead.gt.0) then
          kphead=1000
        end if

        if(transport.and.(kpconc.gt.0)) then
          kpconc=1000
        end if

        if(kpvel.gt.0) then
          kpvel=1000
        end if

      end if
c
c  ...send info to units 40 and 41
c
      if(kphead.gt.0) then
          write(40) nx,ny,nz,nln,kgrid
      end if
      if(kpconc.gt.0) then
        write(41) nx,ny,nz,nln,kgrid
c        write(81,*) nx,ny,nz,nln,kgrid
      end if

c************************************************************************
c   Group 4: Grid data
c************************************************************************
 
      write(*,*) ' Reading GROUP 4'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 4'
         stop
      endif
c
c
c     if gen_grid is false (i.e. kgrid=0)
c     use already generated grid (either 2D or 3D)
c       read iwhole
c         iwhole=0 --> read the nodal coordinates and incidences
c                      for one 2d slice (from ascii files 50 and 51)
c                      and subsequently generate the whole grid.
c         iwhole=1 --> read the nodal coordinates and incidences
c                      for the whole grid from binary files 11 and 12.
c         iwhole=2 --> read the nodal coordinates and incidences
c                      for the whole grid from ASCII files 11 and 12
c     
      if(kgrid.eq.0) then
        read(55,*) iwhole
        if(kphead.gt.0) write(40) iwhole
        if(kpconc.gt.0) then
          write(41) iwhole
c          write(81,*) iwhole
        endif
        if(iwhole.eq.0) then
          open(11,file=prefix(:lenprefix)//name(:lenname)//'.nod',
     &              status='unknown')
          rewind(11)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &                    name(:lenname)//'.nod'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.nod'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.nod'
          endif
          open(12,file=prefix(:lenprefix)//name(:lenname)//'.inc',
     &              status='unknown')
          rewind(12)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &                name(:lenname)//'.inc'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.inc'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.inc'
          endif
c
c  ...read nodal coordinates for one slice
c    
c     can use different formats depending on 3d element used
c
          read(11,*) nndsl
          read(11,*) (x(i),y(i),i=1,nndsl)
          read(11,*) nz
          read(11,*) (zi(i),i=1,nz)
          nn=nndsl*nz
c
c  ...generate nodal coordinates for whole grid
c    
          do 702 ns=1,nz
            iref=nndsl*(ns-1)
            do 701 nyo=1,nndsl
              ini=nyo+iref
              x(ini)=x(nyo)
              y(ini)=y(nyo)
              z(ini)=zi(ns)
701         continue
702       continue
c
c  ...read element incidences for one slice
c    
          if(nln.eq.8) then  
            read(12,*) nesl
            read(12,*) ((in(j,i),j=1,nln2),i=1,nesl)
          elseif(nln.eq.6) then
            read(12,*) nesl
            read(12,*) (in(1,k),in(2,k),in(3,k),k=1,nesl)
            nesl=nesl*(nz-1)
          end if
c
c  ...generate element incidences for whole grid
c    
          do 272 i=1,nesl
            do 271 j=1,nln2
              in(nln2+j,i)=in(j,i)+nndsl
271         continue
272       continue
          if(nz.ge.3) then
            do 274 ns=2,nz-1
              do 273 nowel=1,nesl
                n=(ns-1)*nesl+nowel
                nold=(ns-2)*nesl+nowel
                do 278 j=1,nln
                  in(j,n)=in(j,nold)+nndsl
  278           continue
273           continue
274         continue
          end if

        elseif(iwhole.eq.1) then
c
c  ...read the whole grid from BINARY file
c    
          open(11,file=prefix(:lenprefix)//name(:lenname)//'.nod',
     &          status='unknown',form='unformatted')
          rewind(11)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &           name(:lenname)//'.nod'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.nod'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.nod'
          endif
          open(12,file=prefix(:lenprefix)//name(:lenname)//'.inc',
     &           status='unknown',form='unformatted')
          rewind(12)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &                  name(:lenname)//'.inc'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.inc'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.inc'
          endif
          read(11) nn,(x(i),y(i),z(i),i=1,nn)
          read(12) ne,((in(j,i),j=1,nln),i=1,ne)

        elseif(iwhole.eq.2) then    
c     
c     ...read the whole grid from ASCII file
c     
          open(11,file=prefix(:lenprefix)//name(:lenname)//'.nod',
     &          status='unknown',form='formatted')
          rewind(11)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &           name(:lenname)//'.nod'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.nod'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.nod'
          endif
          open(12,file=prefix(:lenprefix)//name(:lenname)//'.inc',
     &           status='unknown',form='formatted')
          rewind(12)
          if(kphead.gt.0) write(40) prefix(:lenprefix)//
     &                  name(:lenname)//'.inc'
          if(kpconc.gt.0) then
            write(41) prefix(:lenprefix)//name(:lenname)//'.inc'
c            write(81,*) prefix(:lenprefix)//name(:lenname)//'.inc'
          endif

          read(11,*) nn
          do 1281 i=1,nn
             read(11,*) x(i),y(i),z(i)
 1281     continue
          read(12,*) ne
cws       changed Sep. 1998
cws       # of elements per slice
          nesl=ne/(nz-1)
cws       # of nodes per slice
          nndsl=nn/nz
cws       end of change
          do 1285 i=1,ne
             read(12,*) (in(j,i),j=1,nln)
 1285     continue

        end if
        write(66,7350) nn,ne,nndsl,nesl
 7350   format(/,9('*')/,'Grid Data',/,9('*'),//,
     &  'Number of nodes:',t40,i8,/,'Number of elements:',t40,i8,/,
     &  'Number of nodes/horizontal slice:',t40,i8,/,
     &  'Number of elements/horizontal slice:',t40,i8)

      elseif(kgrid.eq.1) then
c
c     if kgrid=1 generate the grid
c      read variable_space
c       false, uniform grid size
c              specify total x,y and z-dimensions
c       true, variable grid size
c             specify coordinate locations for x,y and z-dimensions
c
        write(66,7350) nn,ne,nndsl,nesl
        read(55,*) variable_space
        if(kphead.gt.0) write(40) variable_space
        if(kpconc.gt.0) then
          write(41) variable_space
c          write(81,*) variable_space
        endif
        if(.not.variable_space) then
c
c  ...xl, yl, zl are grid lengths in x-, y- and z-directions
c    
          read(55,*) xl,yl,zl
          if(kphead.gt.0) write(40) xl,yl,zl
          if(kpconc.gt.0) then
            write(41) xl,yl,zl
c            write(81,*) xl,yl,zl
          endif
          write(66,4243) xl,yl,zl
 4243     format(/,'Grid dimensions (uniform spacing)',/,33('-'),
     &       /,'xl =',f14.5,
     &       /,'yl =',f14.5,
     &       /,'zl =',f14.5)
          delx=xl/dble(nx-1)
          dely=yl/dble(ny-1)
          delz=zl/dble(nz-1)
          do 165 i=1,nx
            xi(i)=dble(i-1)*delx
  165     continue
          do 166 i=1,ny
            yi(i)=dble(i-1)*dely
  166     continue
          do 167 i=1,nz
            zi(i)=dble(i-1)*delz
  167     continue
        elseif(variable_space) then
          read(55,*) (xi(i),i=1,nx)
          read(55,*) (yi(i),i=1,ny)
          read(55,*) (zi(i),i=1,nz)         
          if(kphead.gt.0) then
            write(40) (xi(i),i=1,nx)
            write(40) (yi(i),i=1,ny)
            write(40) (zi(i),i=1,nz)
          endif
          if(kpconc.gt.0) then
            write(41) (xi(i),i=1,nx)
            write(41) (yi(i),i=1,ny)
            write(41) (zi(i),i=1,nz)
c            write(81,*) (xi(i),i=1,nx)
c            write(81,*) (yi(i),i=1,ny)
c            write(81,*) (zi(i),i=1,nz)
          endif
c
          write(66,4244) 
 4244     format(/,'x-y-z grid coordinates',
     &    ' (variable spacing)',/,41('-'))
 
        end if
c  ...echo back
        write(66,4246)
 4246   format(/,'x-coordinates')
        write(66,4249) (xi(i),i=1,nx)
        write(66,4247)
 4247   format('y-coordinates')
        write(66,4249) (yi(i),i=1,ny)
        write(66,4248)
 4248   format('z-coordinates')
        write(66,4249) (zi(i),i=1,nz)
 4249   format(6f13.5)
        call gridgen
      endif

c************************************************************************
c   Group 5: Physical flow parameters
c************************************************************************
 
c  ...first read k_rand
c            =false  generate properties for zones of elements
c                    defined either with element numbers or 
c                    nodal coordinates
c            =true   k's  are read from random generator
c
      write(*,*) ' Reading GROUP 5'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 5'
         stop
      endif
      read(55,*) k_rand,k_surfer
c
c     set iprop:   material property identifier for each element
c
      do 261 i=1,ne
        iprop(i)=0
  261 continue
      write(66,1245)
 1245 format(//24('*')/,'Physical Flow Parameters',/,24('*'))
c
c     input for non-random hydraulic conductivity
c
      if((.not.k_rand).and.(.not.k_surfer)) then
c
c  ...read flag specifying how zones are defined
c     use_coord = .true. use nodal coordinates
c               = .false. use node number
c
c     when coordinates are used, they should be the coordinates defining
c     the whole extent of the elements defining a specific zone.
c
        read(55,*) use_coord
        if(use_coord) then
c
c  ...read the number of different zones
c
          read(55,*) nzones_prop
c
c  ...if only one zone, then assign same properties to all the elements
c
          if(nzones_prop.eq.1) then
            read(55,*) xfrom_el,xto_el,yfrom_el,yto_el,zfrom_el,zto_el
            read(55,*) dkxx(1),dkyy(1),dkzz(1),stor(1),por(1)
            do 2135 iel=1,ne
              iprop(iel)=nzones_prop
 2135       continue
            write(66,2106)
 2106       format(/'(Constant properties for the whole domain)')
            write(66,2115) nzones_prop
 2115       format(/,'Zone',i4,
     &      '    xfrom      xto    yfrom      yto    zfrom      ',
     &      'zto',/,12x,
     &      '-----      ---    -----      ---    -----      ---')
            write(66,2116) xfrom_el,xto_el,yfrom_el,yto_el,
     &      zfrom_el,zto_el
 2116       format(8x,6(1x,f8.3))
            write(66,2118) dkxx(1),dkyy(1),dkzz(1),stor(1),por(1)
            write(66,2124) ne 
 2118       format(/,'Kxx:',t45,d12.5,/,'Kyy:',t45,d12.5,/,
     &       'Kzz:',t45,d12.5,/,'Specific storage coefficient:',
     &       t45,d12.5,/,'Porosity:',t45,d12.5)
 2124       format(/'Number of elements having above properties:',i7)
          elseif(nzones_prop.gt.1) then
c
c  ...if more than one zone, repeat for ech zone the following
c
            numelem=0
            do 65 izone=1,nzones_prop
              numalreadydefined=0
              read(55,*) xfrom_el,xto_el,yfrom_el,yto_el,
     &                   zfrom_el,zto_el
              read(55,*) ckx,cky,ckz,ss,porb
              call felem(izone,ncount,xfrom_el,xto_el,
     &          yfrom_el,yto_el,zfrom_el,zto_el,numalreadydefined)
              dkxx(izone)=ckx
              dkyy(izone)=cky
              dkzz(izone)=ckz
              stor(izone)=ss
              por(izone)=porb
              write(66,2115) izone
              write(66,2116) xfrom_el,xto_el,yfrom_el,yto_el,
     &                       zfrom_el,zto_el
              write(66,2118) ckx,cky,ckz,ss,porb
              write(66,2124) ncount
              if (numalreadydefined.gt.0) write(66,*) ' WARNING: '
     $             ,numalreadydefined
     $             ,' nodes in this zone were already defined', 
     $             ' hope thats ok'
              numelem=numelem+ncount-numalreadydefined
   65       continue
            if(numelem.ne.ne) then
              write(66,4096) numelem,ne
              write(*,4096) numelem,ne
 4096         format(//'*** ERROR when inputting zones of physical ',
     &        'properties using coordinates ***',/,'The number of',
     &        ' elements specified is',i7,', and the total number of ',
     &        'elements is',i7,//,'Program stopped')
              stop
            end if
          end if
        elseif(.not.use_coord) then
c
c  ...if coordinates are not used, specify element ranges
c
          read(55,*) nzones_prop
          numelem=0
          do 55 izone=1,nzones_prop
            numalreadydefined=0
            read(55,*) i1,i2,ckx,cky,ckz,ss,porb
            do 53 j=i1,i2
               if(iprop(j).ne.0) then numalreadydefined
     $              =numalreadydefined+1
              iprop(j)=izone  
   53       continue
            dkxx(izone)=ckx
            dkyy(izone)=cky
            dkzz(izone)=ckz
            stor(izone)=ss
            numelem=numelem+abs(i2-i1+1)-numalreadydefined
            write(66,3010) i1,i2,ss,ckx,cky,ckz,porb
 3010       format(/'Physical properties for elements',i6,' to',
     &      i6,/,47('-'),/,'Specific storage coefficient:',t35,
     &      d12.5,/,'Kxx:',t35,d12.5,/,'Kyy:',t35,d12.5,
     &           /,'Kzz:',t35,d12.5,/,'Porosity:',t35,d12.5,/)
            if (numalreadydefined.gt.0) write(66,*) ' WARNING: '
     $           ,numalreadydefined
     $           ,' nodes in this zone were already defined',
     &           ' hope thats ok'
   55     continue
          if(numelem.ne.ne) then
            write(66,4093) numelem,ne
            write(*,4093) numelem,ne
 4093       format(//'*** ERROR when inputting zones of physical ',
     &      'properties using node numbers ***',/,'The number of',
     &      ' elements specified is',i7,', and the total number of ',
     &      'elements is',i7,//,'Program stopped')
            stop
          end if
        end if
      elseif(k_rand) then
c
c  ...check if the maximum number of property zones is 
c     equal to maxne. otherwise the program stops a
c     compilation is required
c
        if(maxkzn.lt.ne) then
          write(*,8460) maxkzn,ne
          write(66,8460) maxkzn,ne
 8460     format('**** ERROR *****',/,'Input from a random ',
     +    'field generator is required',/,'but the maximum ',/,
     +    'number of k zones is',i7,' which is less',/,
     +    'than the number of elements',i7,/,'A new compilation is ',
     +    'required, *** Program stopped ***')
          stop
        end if
c
c  ...random generator input
c
c     read the file name for hydraulic conductivities
c
        nzones_prop=1

        open(unit=25,file=prefix(:lenprefix)//name(:lenname)//'.gen',
     &       status='old',form='unformatted')
        rewind(25)
c
c  ...read specific storage ss
c     read conversion factor convf for hydraulic conductivities
c     coming from generator (to convert for possibly different units)
c
        read(55,*) ss,porb
        read(55,*) convf
        read(55,*) vert
c
        write(66,3012) ss,porb,convf,vert,
     &                 prefix(:lenprefix)//name(:lenname)//'.gen'
 3012   format(/,'Specific storage coefficient:',t35,d16.9,/,
     &          'Porosity:',t35,d16.9,/,
     &          'Conversion factor:',t35,d16.9,/,
     &          'Ratio hor/ver :',t35,d16.9,/,
     &          'Hydraulic conductivities read from: ',a32)
        read(25) k1,k2,k3,ak1,ak2,ak3
        iel=0
        write(66,3013) k1,k2,k3,ak1,ak2,ak3
 3013   format(/,5x,'k1:',t15,i6,/,5x,'k2:',t15,i6,
     &   /,5x,'k3:',t15,i6,/,5x,'ak1:',t11,f10.3,/,5x,
     &   'ak2:',t11,f10.3,/,5x,'ak3:',t11,f10.3/)
        write(66,3027)
 3027   format('*** This program reads the first variable coming',/,
     &  'from fgen.f and assumes that it is ln(hydraulic conductivity')
        ckmax=-1.0d+20
        ckmin=1.0d+20
        do 3479 i=1,nz-1
          do 3478 j=1,ny-1
            do 3477 k=1,nx-1
              iel=iel+1
              read(25) var1,var2
              dkxx(iel)=dble(var1)
              if(iel.le.100) then
                write(66,3017) iel,var1,dkxx(iel)
              end if
              dkxx(iel)=convf*dexp(dkxx(iel))
              if(dkxx(iel).gt.ckmax) ckmax=dkxx(iel)
              if(dkxx(iel).lt.ckmin) ckmin=dkxx(iel)
              if(iel.le.100) then
                write(66,3018) dkxx(iel)
              end if
 3477       continue
 3478     continue
 3479   continue
 3017   format(/3x,'Element:',i7,2x,'var1:',e16.9,'kxx:',d16.9)
 3018   format(10x,'kxx (converted):',d16.9)
        write(66,3024) ckmax,ckmin
 3024   format('Maximum k-value: ',d15.9,' minimum k-value: ',d15.9)
        do 455 l=1,ne
c         stor(l)=ss
          iprop(l)=1
          dkyy(l)=dkxx(l)
          dkzz(l)=dkxx(l)/vert
  455   continue
        stor(1)=ss
        por(1)=porb

      elseif(k_surfer)then

        nzones_prop=1
c
c  ...check if the maximum number of property zones is
c     equal to maxne. otherwise the program stops a
c     compilation is required
c
        if(maxkzn.lt.ne) then
          write(66,9460) maxkzn,ne
          write(*,9460) maxkzn,ne
 9460     format('**** ERROR *****',/,'Input from a random ',
     +    'field generator is required',/,'but the maximum ',/,
     +    'number of k zones is',i7,' which is less',/,
     +    'than the number of elements',i7,/,'A new compilation is ',
     +    'required, *** Program stopped ***')
          stop
        end if
c
c     read the file name for horizontal hydraulic conductivities
c
 20     format(a40)

        iel=0

        do 7479 iz=1,nz-1
          read(55,*) convf,ss,porb
          read(55,20) surffile
          write(66,*) ' Reading kfh layer',iz,' from file ',surffile
          write (66,7788) convf,ss,porb
 7788     format ('Conversion factor:   ',d16.9,/,
     &            'specific storage:   ',d16.9,/,
     &            'Porosity:   ',d16.9,/)
         lensurf=index(surffile,' ') - 1
          open(unit=25,file=prefix(:lenprefix)//surffile(:lensurf),
     &         status='old', recl=5000)

          read(25,10) grtitle
          read (25,*) ix,iy
          if ((ix.ne.nx-1).or.(iy.ne.ny-1)) then
            write(66,*) ' ERROR in file ',surffile
            write(66,*) '  Number of elements in file  X ',ix,' Y ',
     &             iy,'  Number of elements in TBC   X ',nx,' Y ',ny
            write(*,*) ' WARNING number of elements in kf-file ,
     &             surffile, is not consistent with given values !'
          endif
          read(25,10) grtitle
          read(25,10) grtitle
          read (25,*) ckmin,ckmax


          do 7478 j=1,ny-1
            iel=iel+1
            read(25,*) (dkxx(k),k=iel,iel+nx-2)
c
c     The hydraulic conductivities in the z-direction (dkzz(l)) will 
c     no longer be
c     divided by the factor 'vert' defined for each layer. It is important
c     that the position and the number of the nodes are the same as
c     those of the first layer.
c

            do 3465 k=iel,iel+nx-2
              dkxx(k)=convf*dkxx(k)
              por(k)=porb
              stor(k)=ss
              dkyy(k)=dkxx(k)
cws              dkzz(k)=dkxx(k)/vert
c              write(66,'(10d12.4)') dkxx(k)

 3465       continue
           iel=iel+nx-2

 7478     continue

cws        do 4343 l=1,ne
cws           write(66,1001) dkxx(l),dkyy(l)
cws 1001      format(10d12.4,/,10d12.4)
cws 4343   continue

          close(25)
          write(66,3024) ckmax,ckmin
 7479   continue
c     read the file name for vertical hydraulic conductivities
c
cws 20     format(a40)

        iel=0

        do 7579 iz=1,nz-1
cws
          read(55,*) convf
          read(55,20) surffile
          write(66,*) ' Reading kfv layer',iz,' from file ',surffile
          write (66,7588) convf
 7588     format ('Conversion factor:   ',d16.9,/)
cws     &            'specific storage:   ',d16.9,/,
cws     &            'Porosity:   ',d16.9,/)
         lensurf=index(surffile,' ') - 1
          open(unit=26,file=prefix(:lenprefix)//surffile(:lensurf),
     &         status='old', recl=5000)

          read(26,10) grtitle
          read (26,*) ix,iy
          if ((ix.ne.nx-1).or.(iy.ne.ny-1)) then
            write(66,*) ' ERROR in file ',surffile
            write(66,*) '  Number of elements in file  X ',ix,' Y ',
     &             iy,'  Number of elements in TBC   X ',nx,' Y ',ny
            write(*,*) ' WARNING number of elements in kf-file ,
     &             surffile, is not consistent with given values !'
          endif
          read(26,10) grtitle
          read(26,10) grtitle
          read (26,*) ckmin,ckmax


          do 7578 j=1,ny-1
            iel=iel+1
            read(26,*) (dkzz(k),k=iel,iel+nx-2)
c
cws
            do 3466 k=iel,iel+nx-2
              dkzz(k)=convf*dkzz(k)
c              write(66,'(10d12.4)') dkzz(k)

 3466       continue
cws        eoc   
c
           iel=iel+nx-2

 7578     continue

cws        do 4344 l=1,ne
cws           write(66,1001) dkzz(l)
cws 1002      format(10d12.4)
cws 4344   continue

          close(26)
          write(66,3024) ckmax,ckmin
 7579   continue
cws   **********************************
cws     incorporate inactive nodes 
cws   **********************************
cws   flag inactive elemts
        do 755 l=1,ne
          inactelem(l)=.false.
          iprop(l)=l
          if ((dkxx(l).le.0).and.(dkyy(l).le.0).and.(dkzz(l).le.0)) 
     &      inactelem(l)=.true.
  755   continue
cws   flag inactive nodes. First set all nodes as inactive
        do 345 i=1,nn
	     inactnode(i)=.true.
  345   continue
cws   Loop over active elements. Reactivate their nodes, 
cws   leaving out inactive nodes
        do 350 i=1,ne
	   if(.not.inactelem(i)) then
            do 348 j=1,nln
               inactnode(in(j,i))=.false.
  348       continue
         endif
  350   continue
      endif
cws      do 346 i=1,nn
cws	   write(66,*) i, inactnode(i)
cws  346 continue

cws1        do 755 l=1,ne
cws1          iprop(l)=l
cws1          if ((dkxx(l).le.0).and.(dkyy(l).le.0).and.(dkzz(l).le.0)) then
cws1             iprop(l)=0
cws1          endif
cws          if (dkxx(l).gt..99e-9.or.dkxx(l).lt.1.001e-9) write (6,*) 'Tr'
cws          if (dkxx(l).gt..99e-9.or.dkxx(l).lt.1.001e-9) 
cws     &        dkxx(l) = 1.e-12
cws          if (dkyy(l).gt..99e-9.or.dkyy(l).lt.1.001e-9) 
cws     &        dkyy(l) = 1.e-12
cws          if (dkzz(l).gt..99e-9.or.dkzz(l).lt.1.001e-9) 
cws     &        dkzz(l) = 1.e-12
cws1          if (dkxx(l).le.0) write (6,*) 'NULL IM kfx-ELEMENT:',l
cws1          if (dkzz(l).le.0) write (6,*) 'NULL IM kfz-ELEMENT:',l
cws1  755   continue
cws      end if
      return
      end




c***************************************************************************

      subroutine rinput2 (lenprefix,prefix,lenname,name)

c***************************************************************************
c
c  ...Read the data for setting up the problem
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      character*80 grtitle
      character*60 prefix,name

      logical boundary_file

      dimension nident(maxnn)

      dsmall=1.0d-6

 10   format(a80)

c************************************************************************
c   Group 6: Initial condition data for flow
c************************************************************************

c
c  ...assign initial conditions for flow
c
c   read default_ic:   true, all nodes have default initial conditions
c                      false, there are nodes with initial conditions
c                      different from the default
c
      write(*,*) ' Reading GROUP 6'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 6'
         stop
      endif
c
c     if initial heads are not read from a file (krestar=0) then
c     read hinit:    initial head within domain
c     if not all nodes have default values, specify them by using zones
c
      if(krestar.eq.0) then
      read(55,*) default_ic
      write(66,8221)
 8221 format(//27('*')/,'Initial Conditions for Flow',/,27('*'),/)
      if(default_ic) write(66,7216)
      if(.not.default_ic) write(66,7218)
 7216 format('All nodes have default initial condition',/)
 7218 format('Default initial condition do not apply to all nodes')
        read(55,*) hinit
        write(66,8238) hinit
 8238   format('Initial hydraulic head within the domain:',d15.8)
        do 71 i=1,nn
           ci(i)=hinit
   71   continue
        if(.not.default_ic)then
c
c     if not all nodes have initial conditions, then
c     first read the number of zones having uniform initial
c     conditions (but different from default).
c
c     for each zone
c       specify its extent with
c       xfrom,xto,yfrom,yto,zfrom,zto
c
c       specify the initial value
c
c  ***note: ninit is recalculated
c
          read(55,*) nzones
          ninit=0
          write(66,8220)
 8220     format(/,'Nodes having initial conditions different from',
     &    ' the default',//,5('  node',
     &      1x,'   value'),/,5('  ----',1x,'   -----'))

          iflag=1
          do 125 i=1,nzones
            ilast=0
            read(55,*) xfrom_nod,xto_nod,yfrom_nod,yto_nod,
     &                 zfrom_nod,zto_nod
            call fnodes(ilast,iend,nident,maxnn,iflag,
     $        xfrom_nod,xto_nod,yfrom_nod,yto_nod,zfrom_nod,zto_nod)
            read(55,*) hzone

            ninit=ninit+iend
c
c  ...check array size for nident()
c
            if(iend.gt.maxnn) then
              write(66,6635) iend, maxnn
              write(*,6635) iend, maxnn
 6635         format(//,80('*'),/,'ERROR when assigning initial ',
     &  'conditions',/,'The number of nodes having non-default initial'
     &  ,' conditions is ',i6,/,'the dimension of nident is maxnn='
     &  ,i6,/,'Recompile by increasing the value of maxnn')
              stop
            end if
            do 123 ii=1,iend
              ci(nident(ii))=hzone
  123       continue
            write(66,1144) (nident(ii),hzone,ii=1,iend)
  125     continue
          write(66,8226) ninit
 8226     format(/'Total number of nodes having different initial',
     &        ' conditions:',i6)

        endif
c
c  ...if restarting simulation, read initial heads from unit 15
c     initial heads are always pressure heads
c     if flag hydhead is set to false, then reset those initial
c     hyraulic heads to pressure heads (for consistency)
c
      elseif(krestar.gt.0) then
        read(15) trestar
        read(15) (ci(i),i=1,nn)
        close(15)
c  ...read the initial time from unit 55(overwrite what was read from 15)
        read(55,*) trestar
        write(66,8223) trestar
 8223   format(//,'Initial heads read from unit 15, (prefix.hin)',/
     &         ,'Initial time: ',d12.7,//)
      end if

c************************************************************************
c   Group 7: First-type boundary conditions for flow
c************************************************************************
 
c
c   read flag
c   dirichlet_bc: true, there are dirichlet nodes
c                 false, no dirichlet nodes
c
      write(*,*) ' Reading GROUP 7'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 7'
         stop
      endif
      read(55,*) dirichlet_bc
      write (66,47) 
   47 format (//39('*'),/,'First-Type Boundary Conditions for Flow',
     & /,39('*'))
      if(dirichlet_bc) write(66,7200)
      if(.not.dirichlet_bc) write(66,7202)
 7200 format(/'There is/are prescribed head node(s)',/)
 7202 format(/'No prescribed head nodes')
c
c  ...initialise flag for first-type nodes
c
      do 45 i=1,nn
        ic(i)=.false.
   45 continue
      if(.not.dirichlet_bc) then
        ndc=0
      elseif(dirichlet_bc) then
c
c  ...first read use_coord
c                =false, input each dirichlet node separately
c                =true , input zone for dirichlet nodes
c                        specify zone by using coordinates
c                        xfrom,xto,yfrom,yto,zfrom,zto
c                        the coordinates must correspond to nodes
c
        read(55,*) use_coord

c     ...read boundary_file
c     =false : input read from standard inputfile name.in
c     nfileid = 55
c     =true  : input read from special file name.dbc
c     nfileid = 99

        read(55,*) boundary_file
        if(.not.boundary_file) nfileid=55
        if(boundary_file) then 
           nfileid=99
           open(unit=99,file=prefix(:lenprefix)//name(:lenname)//'.dbc',
     &          status='unknown')
        endif


        if(.not.use_coord) then
c
c     if use_coord is false, then read
c        jm(i)    = first-type node number
c        hbc1(i) = prescribed head value
c
          read(nfileid,*) ndc
          do 1141 i=1,ndc
             read(nfileid,*) jm(i),hbc1(i)
            ic(jm(i))=.true.
 1141     continue
        elseif(use_coord) then
c
c     if use_coord is true, then read
c      nzones: number of different zones
c      for each zone read:
c      -------------------
c      range of coordinates (x,y,z) having imposed head equal to value
c        (xfrom,xto,yfrom,yto,zfrom,zto)
c      value: imposed value for each node in the zone
c
           read(nfileid,*) nzones
          ndc=0
          ilast=0
          iflag=4
          do 245 i=1,nzones
c
c  ...read zone limits check where dirichlet nodes are located
c   
             if(boundary_file) then
                read(nfileid,*) xfrom_nod,xto_nod,yfrom_nod,yto_nod,
     &               zfrom_nod,zto_nod,hpres
             elseif(.not.boundary_file) then 
                read(nfileid,*) xfrom_nod,xto_nod,yfrom_nod,yto_nod,
     &               zfrom_nod,zto_nod
             endif
 
            call fnodes(ilast,iend,nident,maxnn,iflag,
     $        xfrom_nod,xto_nod,yfrom_nod,yto_nod,zfrom_nod,zto_nod)
c
c  ...check array size for nident()
c
            if(iend.gt.maxnn) then
              write(66,6637) iend, maxnn
              write(*,6637) iend, maxnn
 6637         format(//,80('*'),/,'ERROR when assigning dirichlet ',
     &  'conditions',/,'The number of dirichlet nodes '
     &  ,'is ',i6,/,'The dimension of nident is maxnn='
     &  ,i6,/,'Recompile by increasing the value of maxnn')
              stop
            end if 

            if(.not.boundary_file) read(nfileid,*) hpres

            do 243 ncount=ilast+1,iend
              node=nident(ncount)
              if(.not.ic(node)) then
                ic(node)=.true.
                ndc=ndc+1
                jm(ndc)=node
                hbc1(ndc)=hpres
              end if
  243       continue
            ilast=iend
  245     continue
        end if
c
c  ...echo back
c
        write(66,3023) ndc
 3023   format('Number of prescribed head nodes:',i5,//,5('  node',
     &      1x,'   value'),/,5('  ----',1x,'   -----'))
        write(66,1144) (jm(i),hbc1(i),i=1,ndc)
 1144   format(5(i6,1x,f8.2))
      end if
      close(99)

 
c************************************************************************
c   Group 8: Second-type boundary conditions for flow
c************************************************************************
 
c
c     for each element having a face with a 2nd-type boundary,
c     read the element number, the face number, the normal darcy influx
c     impinging on the face. the convention for
c     numbering faces is given in the comment statements near the
c     beginning of the code. if an element has more than one face that
c     is of the second-type boundary condition, then repeat the element
c     number, face number, flux and concentration for each 2nd-type face.
c
c     if not a dirichlet (first-type) or second-type boundary,
c     the boundary defaults to a second-type with zero normal gradient
c
c   read flux_bc:  true, there are second-type boundary element faces
c                  false, no second-type boundary element faces
      write(*,*) ' Reading GROUP 8'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 8'
         stop
      endif
      read(55,*) flux_bc
      write(66,5358)
 5358 format(//40('*')/,'Second-Type Boundary Conditions for Flow',/,
     &   40('*'),/)
      if(flux_bc) write(66,7204)
      if(.not.flux_bc) write(66,7206)
 7204 format('There is/are second-type element face(s)')
 7206 format('No second-type element faces ')
c
      if(.not.flux_bc) then
         nbc2=0
       elseif(flux_bc) then
        do 3560 i=1,nn
           workvec(i)=.false.
           gb(i)=0.0d0
 3560   continue
cws     Jan 1999
cws     Initialize matrix for saving the 2nd type fluxes
        do 3561 i = 1,ne
           flux_n(i) = 0.0d0
 3561   continue
cws     Jan 1999
        nbc2=0
        nbc2_face=0
cws     changed Sep. 1998
        read(55,*) use_coord
cws     eoc
        if(use_coord) then
c
c        ...Use coordinates to specify zones
c        nzones: number of different zones
c        for each zone
c           x, y and z extent (in terms of element faces)
c           numplane : face number on which 2nd-type b.c. applies (see
c                   convention)
c           flux_norm : value of the fluid flux normal to the face
c
          read(55,*) nzones
          do 5373 izone=1,nzones
            read(55,*) xfrom_el,xto_el,yfrom_el,yto_el,zfrom_el,zto_el
            read(55,*) num_plane,flux_norm
            call fbc2(xfrom_el,xto_el,
     &           yfrom_el,yto_el,zfrom_el,zto_el,num_plane,flux_norm)
 5373     continue
  
cws       changed Sep. 1998
         elseif(.not.use_coord) then
          read(55,*) boundary_file
          if(.not.boundary_file) nfileid=55
          if(boundary_file) then
            nfileid=98
           open(unit=98,file=prefix(:lenprefix)//name(:lenname)//'.2tb',
     &          status='unknown')
          endif
cws       read # of elements with 2type bc
          read(nfileid,*) nofelem
          do 1142 i=1,nofelem
             read(nfileid,*) iel,num_plane,flux_norm
cws          iel: element on which 2t bc applies to
cws          do not assign recharge to inactive elements
             if(.not.inactelem(iel)) then
cws          change Jan. 1999
cws          save 2nd type boundary flux in matrix for calculation 
cws          of 3rd type fluxes in transport module
                flux_n(iel) = flux_norm
cws	  
                call bc2(iel,flux_norm,num_plane)
             endif
 1142     continue
        endif
cws     eoc
        do 3980 i=1,nn
          if(workvec(i)) then
            nbc2=nbc2+1
            jbc2(nbc2)=i
            fluxbc2(nbc2)=gb(i)
          end if
 3980   continue
c
c  ...echo back
c
cws      change Sep. 1998
         if(use_coord) then
           write(66,5356) nbc2_face,nbc2
 5356      format(//,'Number of 2nd-type faces:',i6,/
     &        ,'Number of nodes having 2nd-type b.c. contribution:',i6,/
     $        /,4(3x,'node',9x,'flux'))
          elseif(.not.use_coord) then
           write(66,5357) nbc2
 5357      format(//,'2nd-type bc is read elementwise',/
     &        ,'Number of nodes having 2nd-type b.c. contribution:',i6,/
     $        /,4(3x,'node',9x,'flux'))
         endif
cws      eoc
         write(66,5355) (jbc2(i),fluxbc2(i),i=1,nbc2)
 5355    format(4(i7,1x,d12.5))
cc     &        /3('element',1x,'face ',7x,'  flux',1x),/)
cc        write(66,5355) (iel_bc2(i),iface_bc2(i),flux_bc2(i),i=1,nbc2)
cc 5355   format(3(i7,i5,1x,d13.6,1x))
c
      endif
	close (98)
      
c************************************************************************
c   Group 9 Fluid source and sink data
c************************************************************************
 
c     must input:
c     iter_flux: false, assume proportional distribution
c                true, iterate to find flux distribution at well
c     qfac: multiplication factor to check convergence when
c           iterating, i.e. max error when computing the well
c           discharge is qfac*discharge (suggest 0.001)
c     nwell : number of different wells
c
c     for each well:
c     --------------
c       flowrate(i): pumping/injection rate of well(i)
c       x, y, z range of coordinates for well
c         (xfrom,xto,yfrom,yto,zfrom,zto)
c
c     computed internally
c
c     jq(i,nwell): node number defining well(i)
c
c   wells:        true, there are injection/withdrawal wells
c                 false, no injection/withdrawal wells
c     
      write(*,*) ' Reading GROUP 9'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 9'
         stop
      endif
      read(55,*) wells
      write(66,6619)
 6619 format(//26('*')/,'Fluid Source and Sink Data',/,26('*'),/)
      if(wells) write(66,7208)
      if(.not.wells) write(66,7210)
 7208 format('There is/are injection/withdrawal well(s)')
 7210 format('No injection/withdrawal well(s)')

      if(.not.wells)then
        nwell=0
        nntot_well=0
      elseif(wells)then
        read(55,*) iter_flux
        read(55,*) qfac
        read(55,*) nwell
        write(66,6126) nwell
        if(iter_flux) write(66,6116) qfac
        if(.not.iter_flux) write(66,6117) qfac
        nntot_well=0

        read(55,*) use_coord
        iflag=2
        if(use_coord) then
          do 6125 i=1,nwell
            ilast=0
c
c ...check where well is located (in terms of node number(s))
c   
            read(55,*) flowrate(i)
            read(55,*) xfrom_nod,xto_nod,yfrom_nod,yto_nod,
     &               zfrom_nod,zto_nod
            call fnodes(ilast,iend,nident,maxnn,iflag,
     $       xfrom_nod,xto_nod,yfrom_nod,yto_nod,zfrom_nod,zto_nod)

c
c  ...check array size for nident()
c
            if(iend.gt.maxnn) then
              write(66,6639) iend, maxnn
               write(*,6639) iend, maxnn
6639         format(//,80('*'),/,'ERROR when assigning ',
     &  'source-sink nodes ',/,'The number of well nodes is ',i6,/,
     &  'The dimension of nident is maxnn='
     &  ,i6,/,'Recompile by increasing the value of maxnn')
              stop
            end if 

            do 2010 j=1,iend
              jq(i,j)=nident(j)
 2010       continue
            nn_well(i)=iend
c
c  ...check dimensions
c
            if(iend.gt.maxnz) then
              write(66,8010) i,iend,maxnz
              write(*,8010) i,iend,maxnz
 8010         format(//'*** Dimensioning error ***',/,
     &        'Number of nodes for well',i3,' is',i5,/,
     &        'Maximum allowable is',i5,//,'Program stopped')
              stop
            end if
c
c  ...echo back
c
            write(66,6127) i,flowrate(i),nn_well(i)
            epsq(i)=dabs(qfac*flowrate(i))
            nntot_well=nntot_well+nn_well(i)
            do 6128 j=1,nn_well(i)
              write(66,6130) jq(i,j),x(jq(i,j)),y(jq(i,j)),z(jq(i,j))
 6128       continue
 6125     continue
        elseif(.not.use_coord) then
c
c  ...Use node number to define wells
c
          do 6135 i=1,nwell
            read(55,*) flowrate(i)
            read(55,*) nn_well(i)
            read(55,*) (jq(i,j),j=1,nn_well(i))
cws         changed sep. 1998
            read(55,*) (jslice(i,j),j=1,nn_well(i))
cws         eoc
            write(66,6127) i,flowrate(i),nn_well(i)
            epsq(i)=dabs(qfac*flowrate(i))
            nntot_well=nntot_well+nn_well(i)
            do 6138 j=1,nn_well(i)
              write(66,6130) jq(i,j),x(jq(i,j)),y(jq(i,j)),z(jq(i,j))
 6138       continue
 6135     continue
        endif
      endif

 6126 format(/,'Number of injection/withdrawal wells:',t37,i4)
 6116 format('- Iteration for flux distribution',/,'qfac:',t30,d12.5)
 6117 format('- No iteration for flux distribution',/,
     &      'qfac:',t30,d12.5,'  (not used)')
 6127 format(//5x,'Data for well:',i3,/,5x,18('-'),//,
     & 5x,'Pumping rate:',t30,d12.5,/,5x,
     & 'Number of well nodes:',t39,i3,//,5x,
     & 'Well nodes',12x,'x-y-z location'/,5x,10('-'),12x,14('-')/)
 6130 format(3x,i8,5x,3f12.3)
 
c
c  ...determine conductivity of the elements surrounding 
c     injection/withdrawal wells
c
c     also flag elements having well nodes in their connections
c     for computing the flux
c
c  ...this is used for specifying flux distribution at the well
c
c  ...it is assumed that the grid is regularly spaced in the
c     z-direction if coordinates are used, i.e. one horizontal 
c     slice of nodes has the same z coordinate.
c
      if(nwell.gt.0) then
        num_well_elem=0
        do 6314 i=1,ne
          workvec2(i)=.false.
 6314   continue
        do 6240 i=1,nwell
c
cws     change Sep. 1998
cws     if coordinates are used:
c         ...locate the range of elements to check according to
c         the z coordinate of the well node 
c
          do 6230 j=1,nn_well(i)
          node=jq(i,j)
            if(use_coord) then
cws     eoc
              zplus=z(node)+dsmall
              zminus=z(node)-dsmall
c
c             ...locate z-level of well node
c
              islice=0
              do 6220 iz=1,nz
                if(zi(iz).gt.zminus.and.zi(iz).lt.zplus) then
                  islice=iz
                  goto 6224
                end if
 6220         continue
 6224         continue
cws              write(6,*) islice
              if(islice.eq.0) then
                write(66,2905) i,j,islice
                stop
              endif
cws          change Sep. 1998
cws          ...if wells are indicated by their nodal indices
             elseif(.not.use_coord) then
              islice=jslice(i,j)
cws	      write(6,*) islice
            endif
cws         eoc
c
c  ...identify elements having well nodes in their incidence array
c     first for elements below slice of nodes iz.
c     (only check the upper half of incidences).
c     don't do it for if most-lower well node is at the lowest level
c     (i.e. z=0.0)
c
            if(islice.ne.1) then
              ielem1=(islice-2)*nesl + 1
              ielem2=(islice-1)*nesl
              dkeq_below(i,j)=0.0d0
              ncount=0
              do 6228 iel=ielem1,ielem2
                do 6226 incidence=nln/2+1,nln
                  if(in(incidence,iel).eq.node) then
                    workvec2(iel)=.true.
                    ncount=ncount+1
cws	              write(6,*) 'iprop:', (iprop(iel))
                    dkeq_below(i,j)=dkeq_below(i,j)+dkxx(iprop(iel))
                    goto 6227
                  end if
 6226           continue
 6227           continue
 6228         continue
              dkeq_below(i,j)=dkeq_below(i,j)/dble(ncount)
            end if
c
c  ...look for upper slice of nodes
c
            if(islice.ne.nz) then
              ielem1=(islice-1)*nesl + 1
              ielem2=islice*nesl
              dkeq_above(i,j)=0.0d0
              ncount=0
              do 6238 iel=ielem1,ielem2
                do 6236 incidence=1,nln/2
                  if(in(incidence,iel).eq.node) then
                    workvec2(iel)=.true.
                    ncount=ncount+1
                    dkeq_above(i,j)=dkeq_above(i,j)+dkxx(iprop(iel))
                    goto 6237
                  end if
 6236           continue
 6237           continue
 6238         continue
              dkeq_above(i,j)=dkeq_above(i,j)/dble(ncount)
            end if
 6230     continue
 6240   continue
 2905   format('**** z-level not found ****',
     &  /,'well: ',i4,' well node:',i4,' islice:',i3,/)
        do 6385 i=1,ne
          if(workvec2(i)) then
            num_well_elem=num_well_elem+1
            jq_elem(num_well_elem)=i
          end if
 6385   continue
      end if

c************************************************************************
c   Group 10: Observation well data
c************************************************************************
 
c     must input:
c
c      obs_wells:    true, there are observation wells
c                    false, no observation wells
c
c     for each well:
c     --------------
c       x, y, z range of coordinates for well
c         (xfrom,xto,yfrom,yto,zfrom,zto)
c
c     computed internally
c
c     jobs(i,nobsw): node numbers defining obs. well(i)
c
      write(*,*) ' Reading GROUP 10'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 10'
         stop
      endif
      read(55,*) obs_wells
      write(66,7336)
 7336 format(//21('*')/,'Observation Well Data',/,21('*'),/)
      if(obs_wells) write(66,7212)
      if(.not.obs_wells) write(66,7214)
 7212 format('There is/are observation well(s)')
 7214 format('No observation well(s)')
      if(.not.obs_wells)then
        nobsw=0
        nntot_obsw=0
      elseif(obs_wells)then
        read(55,*) use_coord
        if (use_coord) then
           read(55,*) nobsw
           open(unit=60,file=prefix(:lenprefix)//name(:lenname)//'o.obs'
     $          ,status='unknown')
           open(unit=63,file=prefix(:lenprefix)//name(:lenname)//'o.cob'
     $          ,status='unknown')
           rewind(60)
           rewind(63)
           write(66,6136) nobsw
           nntot_obsw=0

           iflag=3
           do 6165 i=1,nobsw
              ilast=0
c
c ...check where well is located (in terms of node number(s))
c   
              read(55,*) xfrom_nod,xto_nod,yfrom_nod,yto_nod,
     &               zfrom_nod,zto_nod
              call fnodes(ilast,iend,nident,maxnn,iflag,
     $            xfrom_nod,xto_nod,yfrom_nod,yto_nod,zfrom_nod,zto_nod)
c
c  ...check array size for nident()
c
              if(iend.gt.maxnn) then
                 write(66,6645) iend, maxnn
                 write(*,6645) iend, maxnn
 6645            format(//,80('*'),/,'ERROR when assigning obs. well',
     &     ' nodes ',/,'The number of well nodes is ',i6,/,
     &     'The dimension of nident is maxnn='
     &     ,i6,/,'Recompile by increasing the value of maxnn')
                 stop
              end if 

              do 2145 j=1,iend
                 jobs(i,j)=nident(j)
 2145         continue
              nn_obsw(i)=iend
c
c  ...check dimensions
c
              if(iend.gt.maxnz) then
                 write(66,8111) i,iend,maxnz
                 write(*,8111) i,iend,maxnz
 8111            format(//'*** Dimensioning error ***',/,
     &                'Number of nodes for obs. well',i3,' is',i5,/,
     &                'Maximum allowable is',i5,//,'Program stopped')
              end if
c
c  ...echo back
c
              write(66,6137) i,nn_obsw(i)
              nntot_obsw=nntot_obsw+nn_obsw(i)
              do 6168 j=1,nn_obsw(i)
                 write(66,6140) jobs(i,j),x(jobs(i,j)),
     &                    y(jobs(i,j)),z(jobs(i,j))
 6168         continue
 6165      continue
        elseif(.not.use_coord) then
           read(55,*) nobsw
           open(unit=60,file=prefix(:lenprefix)//name(:lenname)//'o.obs'
     $          ,status='unknown')
           open(unit=63,file=prefix(:lenprefix)//name(:lenname)//'o.cob'
     $          ,status='unknown')
           rewind(60)
           rewind(63)
           write(66,6136) nobsw
           nntot_obsw=0
           
           do 10000 i=1,nobsw
              read(55,*) nn_obsw(i)
              read(55,*) (jobs(i,j),j=1,nn_obsw(i))
c
c  ...check dimensions
c
              if(nn_obsw(i).gt.maxnz) then
                 write(66,8111) i,nn_obsw(i),maxnz
                 write(*,8111) i,nn_obsw(i),maxnz
              end if

              write(66,6137) i,nn_obsw(i)
              nntot_obsw=nntot_obsw+nn_obsw(i)
              do 10002 j=1,nn_obsw(i)
                 write(66,6140) jobs(i,j),x(jobs(i,j)),
     &                    y(jobs(i,j)),z(jobs(i,j))
10002         continue

10000      continue
           
        end if   
      endif
 6136 format(/,'Number of observation wells:',t30,i5)
 6137 format(//5x,'Data for observation well:',i3,/,5x,30('-'),//,
     & 5x,'Number of nodes:',t39,i3,//,5x,
     & 'Well nodes',12x,'x-y-z location'/,5x,10('-'),12x,14('-')/)
 6140 format(3x,i8,5x,3f10.3)


c************************************************************************
c   Group 11: Time specifications
c************************************************************************

c  enter
c
c    steady_state: true, steady-state simulation
c                  false, transient
c
c  ...If steady-state flow is chosen and a transport simulation
c     is also performed, times must be specified in the input
c     section for transport. Otherwise, there is no need to
c     specify times.
c
c     tw: time weighting
c     istep: =0 time values provided
c               if different from zero, generate the nts time values
c            =1 constant time step supplied, tmin and tmax
c                                time values are generated
c            =2 variable time step,  enter the following:
c                     tmin:  initial time value
c                     delta: initial time step;
c                     tinc:  time step multiplier
c                     dtmax: maximum time step size
c               time values are then generated
c
c        the above time values will be target time values if time step
c        control is used
c
c     if restarting from previous head values, the initial time
c     has been read in trestar.
c
c     if time_step_control is true then:
c       read delta , (initial time step)
c            control_head  (logical)
c            dhead_allowed (real)
c
c       the times specified above will become target times for the simulation
c
      write(*,*) ' Reading GROUP 11'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 11'
         stop
      endif
      read(55,*) steady_state
      if(steady_state) then
        if(.not.transport) then
          nts=1
c  ...Set a default value for delta.
          delta=1.0d0
          target_time(nts)=delta
        end if
        control_head=.false.
        tw=1.0d0
        twratio=0.0d0
c  ...Reset some printing flags
        if(kphead.gt.0) kphead=1
        if(kpvel.gt.0) kpvel=1
      else
        read(55,*) nts
c
c  ...check dimensions
c
        if(nts.gt.maxnt) then
          write(66,*) 'Dimensioning ERROR'
          write(*,*) 'Dimensioning ERROR'
          write(66,*) ' nts=',nts,' maxnt=',maxnt
          write(*,*) ' nts=',nts,' maxnt=',maxnt
          stop
        end if
        read(55,*) tw,istep
        if(istep.eq.0) then
          read(55,*) (target_time(i),i=1,nts)
          delta=target_time(1)
          if(krestar.eq.1) delta=target_time(1)-trestar
        else
          if(istep.eq.1) then
            read(55,*) tmin, delta
            tinc=1.0d0
            dtmax=delta
          elseif(istep.eq.2) then
            read(55,*) tmin, delta, tinc,dtmax
          end if
c
c  ...generate the time values if necessary (istep.ne.0)
c
          target_time(1)=tmin+delta
          if(krestar.eq.1) target_time(1)=trestar+delta
          deltan=delta
          if(deltan.gt.dtmax) deltan=dtmax
          do 216 i=1,nts-1
            deltan=deltan*tinc
            if(deltan.gt.dtmax) deltan=dtmax
            target_time(i+1)=target_time(i)+deltan
  216     continue
        end if
        twmin=1.d0-tw
        twratio=-1.0d0*twmin/tw

        if(time_step_control) then
          read(55,*) delta
          read(55,*) control_head
          read(55,*) dhead_allowed
          t(1)=delta
          if(krestar.eq.1) t(1)=trestar+delta
          if(.not.transport) then
            control_conc=.false.
            if(.not.control_head) then
              write(66,*) 'ERROR for time step control'
              write(*,*) 'ERROR for time step control, see output file'
              write(66,*) 'Head controls is false but must be true'
              stop
            end if
          end if
        end if

      end if
c
c  ...echo back time values
c
      write(66,2398)
 2398 format(//19('*')/,'Time Specifications',/,19('*'),/)
      if(steady_state) write(66,7383)
 7383 format('Steady-state flow simulation',/)
      if(.not.steady_state)  then
        write(66,7387)
        write(66,2388) nts
        write(66,2399) (target_time(i),i=1,nts)
      end if
      if(time_step_control.and..not.steady_state) then
        write(66,2450) delta
 2450   format(/,'Time step control **above values are target times**',
     &   /,'Initial time step:',t32,d12.5)
        if(control_head) then
          write(66,2407) dhead_allowed
 2407     format('Desired change in pressure:',t32,d12.5)
        end if
      end if
 7387 format('Transient flow simulation')
 2388 format('Number of time steps:',i5,/,'Time values:')
 2399 format(5(d12.6,2x))
c
c  ...write time info to unit 40
c
      if(kphead.gt.0) then
        if(steady_state) then
          ntstemp=1
          write(40) ntstemp,kphead,target_time(1)
        else
          write(40) nts,kphead,(target_time(i),i=1,nts)
        end if
      end if
      if(.not.steady_state.and.kpconc.gt.0) then
        write(41) nts,kpconc,(target_time(i),i=1,nts)
c        write(81,*) nts,kpconc,(target_time(i),i=1,nts)
      end if

c************************************************************************
c   Group 12: ORTHOMIN solver data for flow
c************************************************************************

c ...read error parameter for ORTHOMIN iterative solution algorithm
c
c    values are required for the following
c       residual error (resid_err)
c       relative error (relat_err)
c       absolute error (absol_err)
c    iterative process is stopped when one of the above criterion is met
c
c       it is recommended to use:
c         relat_err = 1.e-4 to 1.e-6 (smaller the better, but
c                     smaller values take more execution time)
c         but make sure the other criteria are low enough (especially make
c         sure that the residual error is low enough) set the flag isolv_info
c         to true to check the iterative process
c
c       when only one criteria needs to be met, set the two other to very
c       small values (e.g. 1.0d-20) so that they don't have any influence
c
c       isolv_info : true to echo iterative behaviour during ORTHOMIN
c                    (useful if strange behaviour appears during
c                    a simulation)
c       check_residual : true to compute residual norm during ORTHOMIN
c
      write(*,*) ' Reading GROUP 12'
      read(55,10) grtitle
      if (grtitle(1:1).ne.'*') then
         write(*,*) 'Error reading group title of group 12'
         stop
      endif
      read(55,*) resid_err,relat_err,absol_err
      read(55,*) isolv_info
      read(55,*) check_residual
      read(55,*) order2
      write(66,7393) resid_err,relat_err,absol_err
 7393 format (//29('*')/,'ORTHOMIN Solver Data for Flow'/,29('*'),//,
     1  'Residual error for ORTHOMIN solver:  ',t37,d15.8/,
     1  'Relative error for ORTHOMIN solver:  ',t37,d15.8/,
     1  'Absolute error for ORTHOMIN solver:  ',t37,d15.8,/)
      if(isolv_info) write(66,7398)
 7398 format('Extra info written during ORTHOMIN iteration')
      if(.not.isolv_info) write(66,7399)
 7399 format('No extra info written during ORTHOMIN iteration')
      if(check_residual) write(66,7435)
 7435 format('Residual is checked during ORTHOMIN iteration')
      if(.not.check_residual) write(66,7437)
 7437 format('Residual is not checked during ORTHOMIN iteration')
      if(order2) write(66,7445)
 7445 format('Second-order factorization')
      if(.not.order2) write(66,7447)
 7447 format('First-order factorization')




      return
      end
 
c***************************************************************************

      subroutine pgrid

c***************************************************************************
c
c  ...Output the grid information if specified
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      if(kpmsh.eq.0) goto 19
      write (66,26)
   26 format (//,'Nodal coordinates'/,17('-')//,
     &  2('  node',9x,'x',9x,'y',9x,'z',2x),
     &  /,2('  ----',9x,'-',9x,'-',9x,'-',2x))
      write (66,22) (i,x(i),y(i),z(i),i=1,nn)
   22 format (2(i6,3(1x,f10.2),2x))
      if(kpmsh.le.1) go to 19
      if(nln.eq.8) then
        write(66,25)
   25   format (/,'3d brick element incidences'/,27('-')/)
        write (66,23) (i,(in(j,i),j=1,nln),i=1,ne)
   23   format (2(i6,2x,8i6))
      else
        write (66,925)
  925   format (/,'3d triangular prism element incidences',
     &          /,38('-')/)
        write (66,923) (i,(in(j,i),j=1,nln),i=1,ne)
  923   format (2(i6,2x,6i6))
      end if
   19 continue

      return
      end

c***************************************************************************

      subroutine check_size
 
c***************************************************************************
c
c  ...Compare the size of problem to the maximum dimensions allowed
c
c***************************************************************************
 
      include 'tbc.prm'
      include 'tbc.dim'
c
      logical pass
      pass=.true.

      if(nx.gt.maxnx) then 
        write(66,1) maxnx,nx
        write(*,1) maxnx,nx
   1    format(/10x,'Dimensioning error: maxnx = ',i7,
     +         /10x,'                       nx = ',i7)
        pass=.false.
      end if
c
      if(ny.gt.maxny) then 
        write(66,2) maxny,ny
        write(*,2) maxny,ny
   2    format(/10x,'Dimensioning error: maxny = ',i7,
     +         /10x,'                       ny = ',i7)
        pass=.false.
      end if
c
      if(nz.gt.maxnz) then 
        write(66,3) maxnz,nz
        write(*,3) maxnz,nz
   3    format(/10x,'Dimensioning error: maxnz = ',i7,
     +         /10x,'                       nz = ',i7)
        pass=.false.
      end if
c
      if(nn.gt.maxnn) then 
        write(66,4) maxnn,nn
        write(*,4) maxnn,nn
    4    format(/10x,'Dimensioning error: maxnn = ',i7,
     +          /10x,'                       nn = ',i7)
        pass=.false.
      end if
c
      if(ne.gt.maxne) then 
        write(66,5) maxne,ne
        write(*,5) maxne,ne
   5    format(/10x,'Dimensioning error: maxne = ',i7,
     +         /10x,'                       ne = ',i7)
        pass=.false.
      end if
c
      if(nln.gt.maxnln) then 
        write(66,6) maxnln,nln
        write(*,6) maxnln,nln
   6    format(/10x,'Dimensioning error: maxnln = ',i7,
     +         /10x,'                       nln = ',i7)
        pass=.false.
      end if
c
      if(ndc.gt.maxndc) then 
        write(66,7) maxndc,ndc
        write(*,7) maxndc,ndc
   7    format(/10x,'Dimensioning error: maxndc = ',i7,
     +         /10x,'                       ndc = ',i7)
        pass=.false.
      end if
c
      if(num_well_elem.gt.maxwelle) then
        write(66,69) maxwelle,num_well_elem
        write(*,69) maxwelle,num_well_elem
  69    format(/10x,'Dimensioning error:      maxwelle = ',i7,
     +         /10x,'                    num_well_elem = ',i7)
        pass=.false.
      end if
c
      if(nwell.gt.maxwell) then 
        write(66,9) maxwell,nwell
        write(*,9) maxwell,nwell
    9    format(/10x,'Dimensioning error: maxwell = ',i7,
     +         /10x,'                       nwell = ',i7)
        pass=.false.
      end if
c
      if(nntot_well.gt.maxwelln) then
        write(66,10) maxwelln,nntot_well
        write(*,10) maxwelln,nntot_well
   10   format(/10x,'Dimensioning error:    maxwelln = ',i7,
     +         /10x,'                          numss = ',i7)
        pass=.false.
      end if
c
      if(nbc2.gt.maxbc2) then 
        write(66,8) maxbc2,nbc2
        write(*,8) maxbc2,nbc2
   8    format(/10x,'Dimensioning error: maxbc2 = ',i7,
     +         /10x,'                      nbc2 = ',i7)
        pass=.false.
      end if
c
      if(nobsw.gt.maxobw) then
        write(66,11) maxobw,nobsw
        write(*,11) maxobw,nobsw
   11   format(/10x,'Dimensioning error: maxobw  = ',i7,
     +         /10x,'                      nobsw = ',i7)
        pass=.false.
      end if
c
      if(nntot_obsw.gt.maxobwn) then
        write(66,12) maxobwn,nntot_obsw
        write(*,12) maxobwn,nntot_obsw
  12    format(/10x,'Dimensioning error: maxobwn = ',i7,
     +               /10x,'           nntot_obsw = ',i7)
        pass=.false.
      end if
c
      if(nts.gt.maxnt) then 
        write(66,13) maxnt,nts
        write(*,13) maxnt,nts
  13    format(/10x,'Dimensioning error: maxnt = ',i7,
     +         /10x,'                      nts = ',i7)
        pass=.false.
      end if
c
      if(nzones_prop.gt.maxpzn) then
        write(66,19) maxpzn,nzones_prop
        write(*,19) maxpzn,nzones_prop
   19    format(/10x,'Dimensioning error: maxpzn = ',i7,
     +          /10x,'               nzones_prop = ',i7)
        pass=.false.
      end if
c
      if(.not.pass) then
        write(*,*) ' *** Dimensioning error *** Program stopped ***'
        stop
      end if
      return
      end
 
c***************************************************************************
 
      subroutine coeffd
 
c***************************************************************************
c
c  ...Initialize the local coefficient matrices for a finite difference mesh
c
c***************************************************************************
 
      include 'tbc.prm'
      include 'tbc.dim'
      dimension cmi(4,4),aax(4,4),aay(4,4),
     *  aaxy(4,4),aayz(4,4),aazx(4,4),aayzh(4,4),aazxh(4,4)

      data cmi/1.d0,0.d0,0.d0,0.d0,0.d0,1.d0,0.d0,0.d0,0.d0,
     &         0.d0,1.d0,0.d0,0.d0,0.d0,0.d0,1.d0/
      data aax/.5d0,-.5d0,0.d0,0.d0,-.5d0,.5d0,0.d0,0.d0,0.d0,
     &         0.d0,.5d0,-.5d0,0.d0,0.d0,-.5d0,.5d0/
      data aay/.5d0,0.d0,0.d0,-.5d0,0.d0,.5d0,-.5d0,0.d0,0.d0,
     &         -.5d0,.5d0,0.d0,-.5d0,0.d0,0.d0,.5d0/
      data aaxy/1.d0,0.d0,-1.d0,0.d0,0.d0,-1.d0,0.d0,1.d0,-1.d0,
     &          0.d0,1.d0,0.d0,0.d0,1.d0,0.d0,-1.d0/
      data aayz/2.d0,1.d0,0.d0,0.d0,1.d0,2.d0,0.d0,0.d0,0.d0,0.d0,
     &          -2.d0,-1.d0,0.d0,0.d0,-1.d0,-2.d0/
      data aazx/2.d0,0.d0,0.d0,1.d0,0.d0,-2.d0,-1.d0,0.d0,0.d0,
     &          -1.d0,-2.d0,0.d0,1.d0,0.d0,0.d0,2.d0/
      data aayzh/0.d0,0.d0,1.d0,2.d0,0.d0,0.d0,2.d0,1.d0,-1.d0,
     &           -2.d0,0.d0,0.d0,-2.d0,-1.d0,0.d0,0.d0/
      data aazxh/0.d0,2.d0,1.d0,0.d0,-2.d0,0.d0,0.d0,-1.d0,-1.d0,
     &           0.d0,0.d0,-2.d0,0.d0,1.d0,2.d0,0.d0/
c
c  ...Adjust the 4*4 matrices
c     cmi,aax,aay are already adjusted
c
      do 5 i=1,4
        do 4 j=1,4
          if(xterms) then
            aaxy(i,j) = aaxy(i,j)*half
            aayz(i,j) = aayz(i,j)*third
            aazx(i,j) = aazx(i,j)*third
            aayzh(i,j) = aayzh(i,j)*third
            aazxh(i,j) = aazxh(i,j)*third
          else
            aaxy(i,j) = 0.0d0
            aayz(i,j) = 0.0d0
            aazx(i,j) = 0.0d0
            aayzh(i,j) = 0.0d0
            aazxh(i,j) = 0.0d0
          end if
    4   continue
    5 continue
c
c  ...Expand the elemental influence coefficient matrices
c
      do 50 i=1,4
        i2 = i+4
        do 40 j=1,4
          j2 = j+4
c
          edxx(i,j) = aax(i,j)
          edxx(i,j2) = 0.0d0
          edxx(i2,j) = 0.0d0
          edxx(i2,j2) = edxx(i,j)
c
          edyy(i,j) = aay(i,j)
          edyy(i,j2) = 0.0d0
          edyy(i2,j) = 0.0d0
          edyy(i2,j2) = edyy(i,j)
c
          edzz(i,j) = cmi(i,j)*half
          edzz(i,j2) = -cmi(i,j)*half
          edzz(i2,j) = edzz(i,j2)
          edzz(i2,j2) = edzz(i,j)

c
          edxy(i,j) = (2.d0*third)*aaxy(i,j)
          edxy(i,j2) = aaxy(i,j)*third
          edxy(i2,j) = aaxy(j,i)*third
          edxy(i2,j2) = edxy(i,j)    
c
          edyz(i,j) = aayz(i,j)*half
          edyz(i,j2) = aayzh(i,j)*half
          edyz(i2,j) = aayzh(j,i)*half
          edyz(i2,j2) = -edyz(i,j)
c
          edxz(i,j) = aazx(i,j)*half
          edxz(i,j2) = aazxh(i,j)*half
          edxz(i2,j) = aazxh(j,i)*half
          edxz(i2,j2) = -edxz(i,j)
c
          eb(i,j) = cmi(i,j)
          eb(i,j2) = 0.d0
          eb(i2,j) = 0.d0
          eb(i2,j2) = eb(i,j)
c
   40   continue
   50 continue
c
      do 60 i=1,4
        do 59 j=1,4
          ec(i,j)=cmi(i,j)
   59   continue
   60 continue

      return
      end

c***************************************************************************
 
      subroutine coeffe
 
c***************************************************************************
c
c  ...Initialize the local coefficient matrices for a finite element mesh
c
c     The node numbering is the following
c
c          4-------3     8-------7
c          |       |     |       |
c          |bottom |     | top   |
c          1-------2     5-------6
c
c***************************************************************************
 
      include 'tbc.prm'
      include 'tbc.dim'
      dimension cmi(4,4),aax(4,4),aay(4,4),
     *  aaxy(4,4),aayz(4,4),aazx(4,4),aayzh(4,4),aazxh(4,4)

      data cmi /4.d0,2.d0,1.d0,2.d0,2.d0,4.d0,2.d0,1.d0,1.d0,2.d0,
     &         4.d0,2.d0,2.d0,1.d0,2.d0,4.d0/
      data aax /2.d0,-2.d0,-1.d0,1.d0,-2.d0,2.d0,1.d0,-1.d0,-1.d0,
     &         1.d0,2.d0,-2.d0, 1.d0,-1.d0,-2.d0,2.d0/
      data aay /2.d0,1.d0,-1.d0,-2.d0,1.d0,2.d0,-2.d0,-1.d0,-1.d0,
     &         -2.d0,2.d0,1.d0,-2.d0,-1.d0,1.d0,2.d0/
      data aaxy /1.d0,0.d0,-1.d0,0.d0,0.d0,-1.d0,0.d0,1.d0,-1.d0,
     &         0.d0,1.d0,0.d0,0.d0,1.d0,0.d0,-1.d0/
      data aayz /2.d0,1.d0,0.d0,0.d0,1.d0,2.d0,0.d0,0.d0,0.d0,0.d0,
     &         -2.d0,-1.d0,0.d0,0.d0,-1.d0,-2.d0/
      data aazx /2.d0,0.d0,0.d0,1.d0,0.d0,-2.d0,-1.d0,0.d0,0.d0,
     &         -1.d0,-2.d0,0.d0,1.d0,0.d0,0.d0,2.d0/
      data aayzh /0.d0,0.d0,1.d0,2.d0,0.d0,0.d0,2.d0,1.d0,-1.d0,
     &          -2.d0,0.d0,0.d0,-2.d0,-1.d0,0.d0,0.d0/
      data aazxh /0.d0,2.d0,1.d0,0.d0,-2.d0,0.d0,0.d0,-1.d0,-1.d0,
     &          0.d0,0.d0,-2.d0,0.d0,1.d0,2.d0,0.d0/
c
c  ...Adjust the 4*4 matrices
c
      do 5 i=1,4
        do 4 j=1,4
          cmi(i,j) = cmi(i,j)*xninth
          aax(i,j) = aax(i,j)*sixth
          aay(i,j) = aay(i,j)*sixth
          aaxy(i,j) = aaxy(i,j)*half
          aayz(i,j) = aayz(i,j)*third
          aazx(i,j) = aazx(i,j)*third
          aayzh(i,j) = aayzh(i,j)*third
          aazxh(i,j) = aazxh(i,j)*third
    4   continue
    5 continue
c
c  ...Expand the elemental influence coefficient matrices
c
      do 50 i=1,4
        i2 = i+4
        do 40 j=1,4
          j2 = j+4
c
          edxx(i,j) = (2.d0*third)*aax(i,j)
          edxx(i,j2) = aax(i,j)*third
          edxx(i2,j) = edxx(i,j2)
          edxx(i2,j2) = edxx(i,j)
c
          edyy(i,j) = (2.d0*third)*aay(i,j)
          edyy(i,j2) = aay(i,j)*third
          edyy(i2,j) = edyy(i,j2)
          edyy(i2,j2) = edyy(i,j)
c
          edzz(i,j) = cmi(i,j)*half
          edzz(i,j2) = -cmi(i,j)*half
          edzz(i2,j) = edzz(i,j2)
          edzz(i2,j2) = edzz(i,j)

c
          edxy(i,j) = (2.d0*third)*aaxy(i,j)
          edxy(i,j2) = aaxy(i,j)*third
          edxy(i2,j) = aaxy(j,i)*third
          edxy(i2,j2) = edxy(i,j)    
c
          edyz(i,j) = aayz(i,j)*half
          edyz(i,j2) = aayzh(i,j)*half
          edyz(i2,j) = aayzh(j,i)*half
          edyz(i2,j2) = -edyz(i,j)
c
          edxz(i,j) = aazx(i,j)*half
          edxz(i,j2) = aazxh(i,j)*half
          edxz(i2,j) = aazxh(j,i)*half
          edxz(i2,j2) = -edxz(i,j)
c
          eb(i,j) = (2.d0*third)*cmi(i,j)
          eb(i,j2) = cmi(i,j)*third
          eb(i2,j) = eb(i,j2)
          eb(i2,j2) = eb(i,j)
c
   40   continue
   50 continue
c
      do 60 i=1,4
        do 59 j=1,4
          ec(i,j)=cmi(i,j)
   59   continue
   60 continue
      return
      end
 
c***************************************************************************
 
      subroutine coefpr(l,delt)
 
c***************************************************************************
c
c  ...Compute the local coefficient matrices for prism elements
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
      dimension cmi(4,4),cmilump(4,4),aax(4,4),aay(4,4),
     *  aaxy(4,4),aayz(4,4),aazx(4,4),
     *  aayzh(4,4),aazxh(4,4),
     *  b(3),g(3)

      data cmi /.5d0, .25d0, .25d0, 0.d0, .25d0, .5d0, .25d0, 0.d0,
     &          .25d0, .25d0, .5d0, 0.0d0, 0.0d0, 0.d0, 0.d0, 0.d0/
      data cmilump /1.0d0, 0.0d0, 0.0d0, 0.0d0, 0.d0, 1.d0, 0.d0,
     &               0.d0, 0.0d0, 0.0d0, 1.0d0, 0.d0, 0.d0, 0.d0,
     &               0.d0, 0.d0/

      nln2=nln/2      
      x2=x(in(2,l))-x(in(1,l))
      x3=x(in(3,l))-x(in(1,l))
      y2=y(in(2,l))-y(in(1,l))
      y3=y(in(3,l))-y(in(1,l))
      delt=(x2*y3-x3*y2)*half
      delt2=2.0d0*delt
      delt2inv=1.0d0/delt2
      b(1)=(y2-y3)*delt2inv
      b(2)=y3*delt2inv
      b(3)=-y2*delt2inv
      g(1)=(x3-x2)*delt2inv
      g(2)=-x3*delt2inv
      g(3)=x2*delt2inv
c
      do 40 i=1,nln2
        do 39 j=1,nln2
          aax(i,j)=b(i)*b(j)
          aay(i,j)=g(i)*g(j)
          aaxy(i,j)=b(i)*g(j) + b(j)*g(i)
          aayz(i,j)=g(i) + g(j)
          aayzh(i,j)=g(i) - g(j)
          aazx(i,j)=b(i) + b(j)
          aazxh(i,j)=b(i)-b(j)
   39   continue
   40 continue
c
c  ...Adjust the 3*3 matrices
c
      do 44 i=1,nln2
        do 43 j=1,nln2
          aayz(i,j) = -aayz(i,j)*third
          aazx(i,j) = -aazx(i,j)*third
          aayzh(i,j) = aayzh(i,j)*third
          aazxh(i,j) = aazxh(i,j)*third
   43   continue
   44 continue



c
c  ...Expand the elemental influence coefficient matrices
c
      if(klump.eq.0) then
        do 50 i=1,nln2
          i2 = i+nln2
          do 45 j=1,nln2
            j2 = j+nln2
c
            edxx(i,j) = (2.d0*third)*aax(i,j)
            edxx(i,j2) = aax(i,j)*third
            edxx(i2,j) = edxx(i,j2)
            edxx(i2,j2) = edxx(i,j)
c
            edyy(i,j) = (2.d0*third)*aay(i,j)
            edyy(i,j2) = aay(i,j)*third
            edyy(i2,j) = edyy(i,j2)
            edyy(i2,j2) = edyy(i,j)
c
            edzz(i,j) = cmi(i,j)*half
            edzz(i,j2) = -cmi(i,j)*half
            edzz(i2,j) = edzz(i,j2)
            edzz(i2,j2) = edzz(i,j)

c
            edxy(i,j) = (2.d0*third)*aaxy(i,j)
            edxy(i,j2) = aaxy(i,j)*third
            edxy(i2,j) = aaxy(j,i)*third
            edxy(i2,j2) = edxy(i,j)    
c
            edyz(i,j) = aayz(i,j)*half
            edyz(i,j2) = aayzh(i,j)*half
            edyz(i2,j) = aayzh(j,i)*half
            edyz(i2,j2) = -edyz(i,j)
c
            edxz(i,j) = aazx(i,j)*half
            edxz(i,j2) = aazxh(i,j)*half
            edxz(i2,j) = aazxh(j,i)*half
            edxz(i2,j2) = -edxz(i,j)
c
            eb(i,j) = (2.d0*third)*cmi(i,j)
            eb(i,j2) = cmi(i,j)*third
            eb(i2,j) = eb(i,j2)
            eb(i2,j2) = eb(i,j)
   45     continue
   50   continue
c
      else
        do 70 i=1,nln2
          i2 = i+nln2
          do 75 j=1,nln2
            j2 = j+nln2
c
            edxx(i,j) = aax(i,j)
            edxx(i,j2) = 0.0d0
            edxx(i2,j) = edxx(i,j2)
            edxx(i2,j2) = edxx(i,j)
c
            edyy(i,j) = aay(i,j)
            edyy(i,j2) = 0.0d0
            edyy(i2,j) = edyy(i,j2)
            edyy(i2,j2) = edyy(i,j)
c
            edzz(i,j) = cmi(i,j)*half
            edzz(i,j2) = -cmilump(i,j)*half
            edzz(i2,j) = edzz(i,j2)
            edzz(i2,j2) = edzz(i,j)

c
            edxy(i,j) = aaxy(i,j)
            edxy(i,j2) = 0.0d0
            edxy(i2,j) = 0.0d0
            edxy(i2,j2) = edxy(i,j)    
c
            edyz(i,j) = 0.0d0
            edyz(i,j2) = 0.0d0
            edyz(i2,j) = 0.0d0
            edyz(i2,j2) = 0.0d0
c
            edxz(i,j) = 0.0d0
            edxz(i,j2) = 0.0d0
            edxz(i2,j) = 0.0d0
            edxz(i2,j2) = 0.0d0
c
            eb(i,j) = (2.d0*third)*cmi(i,j)
            eb(i,j2) = cmilump(i,j)*third
            eb(i2,j) = eb(i,j2)
            eb(i2,j2) = eb(i,j)
   75     continue
   70   continue
      end if
c
      do 60 i=1,nln2
        do 59 j=1,nln2
          ec(i,j)=cmi(i,j)
   59   continue
   60 continue

      return
      end

c***************************************************************************
 
      subroutine gridgen

c***************************************************************************
c
c  ...Set up the the grid coordinates and the element incidences
c
c***************************************************************************
 
      include 'tbc.prm'
      include 'tbc.dim'
c
c  ...Determine nodal coordinates
c
      nnode=0
      do 12 k=1,nz
        zcur=zi(k)
        do 11 j=1,ny
          ycur=yi(j)
          do 10 i=1,nx
            xcur=xi(i)
            nnode=nnode+1
            x(nnode)=xcur
            y(nnode)=ycur
            z(nnode)=zcur
   10     continue
   11   continue
   12 continue
c
c  ...Determine elemental incidences
c
      nxy=nx*ny
      nx1=nx+1
      nnode=0
      ielt=0
      do 20 k=1,(nz-1)
        do 30 j=1,(ny-1)
          do 40 i=1,(nx-1)
            if(nln.eq.8) then
              ielt=ielt+1
              nnode=nnode+1
              in(1,ielt)=nnode
              in(2,ielt)=nnode+1
              in(3,ielt)=nnode+nx1
              in(4,ielt)=nnode+nx
              in(5,ielt)=nnode+nxy
              in(6,ielt)=nnode+nxy+1
              in(7,ielt)=nnode+nxy+nx1
              in(8,ielt)=nnode+nxy+nx
            else
              ielt=ielt+1
              nnode=nnode+1
              in(1,ielt)=nnode
              in(2,ielt)=nnode+1
              in(3,ielt)=nnode+nx
              in(4,ielt)=in(1,ielt)+nxy
              in(5,ielt)=in(2,ielt)+nxy
              in(6,ielt)=in(3,ielt)+nxy
c
              ielt=ielt+1
              in(1,ielt)=nnode+1
              in(2,ielt)=nnode+nx1
              in(3,ielt)=nnode+nx
              in(4,ielt)=in(1,ielt)+nxy
              in(5,ielt)=in(2,ielt)+nxy
              in(6,ielt)=in(3,ielt)+nxy
            end if
   40     continue
          nnode=nnode+1
   30   continue
        nnode=nnode+nx
   20 continue
      return
      end
 
c**************************************************************************

      subroutine elemdim(l,dx,dy,dz)

c**************************************************************************
c
c  ...Compute elemental dimensions
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      if(kgrid.eq.1)then
        dx=x(in(2,l))-x(in(1,l))
        dy=y(in(4,l))-y(in(1,l))
        dz=z(in(8,l))-z(in(4,l))
      elseif(kgrid.eq.0)then
        el12=dlac(in(2,l),in(1,l))
        el34=dlac(in(3,l),in(4,l))
        el56=dlac(in(5,l),in(6,l))
        el78=dlac(in(7,l),in(8,l))
        dx=0.25d0*(el12+el34+el56+el78)
        el14=dlac(in(1,l),in(4,l))
        el23=dlac(in(2,l),in(3,l))
        el58=dlac(in(5,l),in(8,l))
        el67=dlac(in(6,l),in(7,l))
        dy=0.25d0*(el14+el23+el58+el67)
        el15=dlac(in(1,l),in(5,l))
        el26=dlac(in(2,l),in(6,l))
        el37=dlac(in(3,l),in(7,l))
        el48=dlac(in(4,l),in(8,l))
        dz=0.25d0*(el15+el26+el37+el48)
      endif

      return
      end

c************************************************************************* 

      double precision function dlac(n1,n2)

c*************************************************************************
c
c  ...Finds the absolute distance between 2 nodes
c
c*************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
      integer*4 n1,n2
      term=(x(n1)-x(n2))**2 + (y(n1)-y(n2))**2 + (z(n1)-z(n2))**2
      dlac=dsqrt(term)
      return
      end

c***************************************************************************

      subroutine bc2(iel,flux_norm,num_plane)

c***************************************************************************
c
c  ...Adjust flux vector for 2nd-type (flux) boundary conditions
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      l=iel
      nface=num_plane
      call farea(nface,area,l,ncount)
      term=area*flux_norm
      do 221 ii=1,ncount
        n1=in(iface(nface,ii,1),l)
cws   change october 1998
cws        write(66,*) n1,inactnode(n1)
cws        if(.not.ic(n1).and..not.inactnode(n1)) then
        if(.not.ic(n1)) then
          workvec(n1)=.true.
          gb(n1)=gb(n1) + term
        end if
  221 continue

      return
      end

c***************************************************************************

      subroutine farea(nface,area,l,ncount)

c***************************************************************************
c
c  ...Find face area
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      ncount=4
      if(nln.eq.8) then
        if(kgrid.eq.1)then
          if(nface.eq.1.or.nface.eq.3) then
c  ...x- and z-lengths of front or back faces
            el1=x(in(6,l))-x(in(5,l))
            el2=z(in(5,l))-z(in(1,l))
          elseif(nface.eq.2.or.nface.eq.4) then
c  ...z- and y-lengths of right or left faces
            el1=z(in(6,l))-z(in(2,l))
            el2=y(in(3,l))-y(in(2,l))
          elseif(nface.eq.5.or.nface.eq.6) then
c  ...x- and y-lenghts of top or bottom faces
            el1=y(in(3,l))-y(in(2,l))
            el2=x(in(2,l))-x(in(1,l))
          endif
        elseif(kgrid.ne.1) then
          if(nface.eq.1)then
            el12=dlac(in(1,l),in(2,l))
            el56=dlac(in(5,l),in(6,l))
            el1=0.5d0*(el12+el56)
            el15=dlac(in(1,l),in(5,l))
            el26=dlac(in(2,l),in(6,l))
            el2=0.5d0*(el15+el26)
          elseif(nface.eq.3)then
            el34=dlac(in(3,l),in(4,l))
            el78=dlac(in(7,l),in(8,l))
            el1=0.5d0*(el34+el78)
            el37=dlac(in(3,l),in(7,l))
            el48=dlac(in(4,l),in(8,l))
            el2=0.5d0*(el37+el48)
          elseif(nface.eq.2)then
            el26=dlac(in(2,l),in(6,l))
            el37=dlac(in(3,l),in(7,l))
            el1=0.5d0*(el26+el37)
            el23=dlac(in(2,l),in(3,l))
            el67=dlac(in(6,l),in(7,l))
            el2=0.5d0*(el23+el67)
          elseif(nface.eq.4)then
            el15=dlac(in(1,l),in(5,l))
            el48=dlac(in(4,l),in(8,l))
            el1=0.5d0*(el15+el48)
            el14=dlac(in(1,l),in(4,l))
            el58=dlac(in(5,l),in(8,l))
            el2=0.5d0*(el14+el58)
          elseif(nface.eq.5)then
            el23=dlac(in(2,l),in(3,l))
            el14=dlac(in(1,l),in(4,l))
            el1=0.5d0*(el14+el23)
            el12=dlac(in(1,l),in(2,l))
            el34=dlac(in(3,l),in(4,l))
            el2=0.5d0*(el12+el34)
          elseif(nface.eq.6)then
            el58=dlac(in(5,l),in(8,l))
            el67=dlac(in(6,l),in(7,l))
            el1=0.5d0*(el58+el67)
            el56=dlac(in(5,l),in(6,l))
            el78=dlac(in(7,l),in(8,l))
            el2=0.5d0*(el56+el78)
          endif
        endif
        area=el1*el2*0.25d0
      elseif(nln.eq.6) then
        if(nface.eq.1.or.nface.eq.2) then
          delt=(x(in(2,l))*y(in(3,l))-x(in(3,l))*y(in(2,l)))*half
          area=delt*third
          ncount=3
        else
          if(nface.eq.3)then
            el1=dlac(in(1,l),in(2,l))
            el2=dlac(in(1,l),in(4,l))
          elseif(nface.eq.4)then
            el1=dlac(in(1,l),in(3,l))
            el2=dlac(in(1,l),in(4,l))
          elseif(nface.eq.5)then
            el1=dlac(in(2,l),in(3,l))
            el2=dlac(in(2,l),in(5,l))
          end if
          area=el1*el2*0.25d0
        end if
      end if

      return
      end

c***************************************************************************

      subroutine fbc2(xfrom_el,xto_el,
     & yfrom_el,yto_el,zfrom_el,zto_el,num_plane,flux_norm)

c***************************************************************************
c
c  ...Search for element faces having 2nd-type b.c. applied to
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      smallpos=1.0d-6
      smallneg=-1.0d-6  
      found=.true.
c
c  ...Locate the z-level
c     It is assumed that the grid is regular in the z-direction
c     but can be irregular in the xy-plane (e.g. if nln=6)
c
      zz1=zfrom_el-smallpos
      zz2=zfrom_el+smallpos
      indez1=0
      do 2000 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez1=iz
          goto 2001 
        end if
 2000 continue
 2001 continue
      if(indez1.eq.0) found=.false.

      zz1=zto_el-smallpos
      zz2=zto_el+smallpos
      indez2=0
      do 2005 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez2=iz
          goto 2006
        end if 
 2005 continue
 2006 continue
      if(indez2.eq.0) found=.false.

      if(.not.found) then
        write(66,3000) indez1,indez2
        write(*,3000) indez1,indez2
        stop
      end if
 3000 format(/,'**** ERROR ****',/,
     &  'Range of elements not found',
     &  /'z-index1=',i4,' z-index2=',i4)

      idum=indez1
      if(indez1.gt.indez2) then
        indez1=indez2
        indez2=idum
      end if
      if(indez1.eq.indez2) then
         indez2=indez1+1
         if (indez1.eq.nz) then
            indez1=nz-1
            indez2=nz
         endif
      end if
c
c  ...If nln=8, it is assumed that the grid is regular in
c     all directions
c
c  ...Loop over elements in the z-range found
c
      if(nln.eq.8) then
        incidencex1=iface(num_plane,1,1)
        incidencex2=iface(num_plane,2,1)
        incidencey1=iface(num_plane,1,1)
        incidencey2=iface(num_plane,4,1)
        if(num_plane.eq.2.or.num_plane.eq.4) then
          incidencey2=iface(num_plane,2,1)
        end if
        do 3050 iz=indez1,indez2-1
          ielem1=(iz-1)*nesl+1
          ielem2=iz*nesl
          do 3030 iel=ielem1,ielem2
cws          change Jan. 1999
cws          save 2nd type boundary flux in matrix for calculation 
cws          of 3rd type fluxes in transport module
             flux_n(iel) = flux_norm
cws	  
            found=.true. 
            x1=x(in(incidencex1,iel))
            x2=x(in(incidencex2,iel))
            y1=y(in(incidencey1,iel))
            y2=y(in(incidencey2,iel))
            check=x1-xfrom_el
            if(check.lt.smallneg) found=.false.
            check=x2-xto_el
            if(check.gt.smallpos) found=.false.
            check=y1-yfrom_el
            if(check.lt.smallneg) found=.false.
            check=y2-yto_el
            if(check.gt.smallpos) found=.false.
            if(found) then
              nbc2_face=nbc2_face+1
              call bc2(iel,flux_norm,num_plane)
            end if
 3030     continue
 3050   continue
      elseif(nln.eq.6) then
        write(66,4000) 
 4000   format('*** WARNING ***',/,'Trying to define 2nd-type',
     &  ' boundary conditions using coordinates',/,'while you are',
     &  ' using triangular prism elements, code needs to be changed',
     &  ' for that',/,'program stopped')
        stop
      end if

      return
      end

c***************************************************************************

      subroutine felem(izone,ncount,xfrom_el,xto_el,
     &  yfrom_el,yto_el,zfrom_el,zto_el,numalreadydefined)

c***************************************************************************
c
c  ...Search for element enclosed in a given property zone
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      smallpos=1.0d-5
      smallneg=-1.0d-5
      found=.true.
c
c  ...Check where desired feature is located (in terms of element number(s))
c
c  ***NOTE***
c     It is assumed that the grid is regular in the z-direction
c     therefore the different z-levels are identical
c     Check the z-range of elements to consider
c
c
      zz1=zfrom_el-smallpos
      zz2=zfrom_el+smallpos
      indez1=0
      do 2000 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez1=iz
          goto 2001 
        end if
 2000 continue
 2001 continue
      if(indez1.eq.0) found=.false.

      zz1=zto_el-smallpos
      zz2=zto_el+smallpos
      indez2=0
      do 2005 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez2=iz
          goto 2006
        end if 
 2005 continue
 2006 continue
      if(indez2.eq.0) found=.false.

      if(.not.found) then
        write(66,3000) indez1,indez2
        write(*,3000) indez1,indez2
        stop
      end if
 3000 format(/,'**** ERROR ****',/,
     &  'Range of elements not found',
     &  /'z-index1=',i4,' z-index2=',i4)

      idum=indez1
      if(indez1.gt.indez2) then
        indez1=indez2
        indez2=idum
      end if
c
c  ...Loop over elements in the z-range found
c
      if(nln.eq.8) then
        ncount=0
        do 3050 iz=indez1,indez2-1
          ielem1=(iz-1)*nesl+1
          ielem2=iz*nesl
          do 3030 iel=ielem1,ielem2
            found=.true. 
            x1=x(in(1,iel))
            x2=x(in(2,iel))
            y1=y(in(1,iel))
            y2=y(in(nln/2,iel))
            check=x1-xfrom_el
            if(check.lt.smallneg) found=.false.
            check=x2-xto_el
            if(check.gt.smallpos) found=.false.
            check=y1-yfrom_el
            if(check.lt.smallneg) found=.false.
            check=y2-yto_el
            if(check.gt.smallpos) found=.false.
            if(found) then
               ncount=ncount+1
               if(iprop(iel).ne.0) numalreadydefined=numalreadydefined+1
              iprop(iel)=izone
            end if
 3030     continue
 3050   continue
      elseif(nln.eq.6) then
c
c  ...Only check the first level of elements in (indez1,indez2)
c     and repeat find for the other levels
c
        ncount=0
        iz=indez1
        ielem1=(iz-1)*nesl+1
        ielem2=iz*nesl
        do 4030 iel=ielem1,ielem2
          found=.true. 
          x1=dmin1(x(in(1,iel)), x(in(2,iel)), x(in(3,iel)) ) 
          x2=dmax1(x(in(1,iel)), x(in(2,iel)), x(in(3,iel)) ) 
          y1=dmin1(y(in(1,iel)), y(in(2,iel)), y(in(3,iel)) ) 
          y2=dmax1(y(in(1,iel)), y(in(2,iel)), y(in(3,iel)) ) 
          check=x1-xfrom_el
          if(check.lt.smallneg) found=.false.
          check=x2-xto_el
          if(check.gt.smallpos) found=.false.
          check=y1-yfrom_el
          if(check.lt.smallneg) found=.false.
          check=y2-yto_el
          if(check.gt.smallpos) found=.false.
          if(found) then
            numberz=indez2-indez1
            do 4020 levelz=1,numberz
              ielement=(levelz-1)*nesl + iel
              ncount=ncount+1
              if(iprop(ielement).ne.0) numalreadydefined
     $             =numalreadydefined+1
              iprop(ielement)=izone
 4020       continue
          end if
 4030   continue
      end if


      return
      end

c***************************************************************************

      subroutine fnodes(ilast,iend,nident,maxdim,iflag,
     &  xfrom_nod,xto_nod,yfrom_nod,yto_nod,zfrom_nod,zto_nod)

c***************************************************************************
c
c  ...Search for nodes given a range of coordinates
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      dimension nident(maxdim)

      dsmall=1.d-5

      found=.true.
c
c  ...Check where desired feature is located (in terms of node number(s))
c     It is assumed that the grid is regular, nodes are numbered
c     first in the x-direction, then the y-direction and finally
c     in the z-direction
c
c  ...Further change will take into account that the xy slices
c     can be generated with a grid builder, therefore not showing
c     any regular xy patterns.
c
      zz1=zfrom_nod-dsmall
      zz2=zfrom_nod+dsmall
      indez1=0

      do 2000 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez1=iz
          goto 2001 
        end if
 2000 continue
 2001 continue
      if(indez1.eq.0) found=.false.
      zz1=zto_nod-dsmall
      zz2=zto_nod+dsmall
      indez2=0
      do 2005 iz=1,nz
        if(zi(iz).ge.zz1.and.zi(iz).le.zz2) then
          indez2=iz
          goto 2006
        end if 
 2005 continue
 2006 continue
      if(indez2.eq.0) found=.false.
c
c  ...Regular grid (8-node brick elements)
c
      if(nln.eq.8) then
        xx1=xfrom_nod-dsmall
        xx2=xfrom_nod+dsmall
        index1=0
        do 2010 ix=1,nx
          if(xi(ix).ge.xx1.and.xi(ix).le.xx2) then
            index1=ix
            goto 2011
          end if
 2010   continue
 2011   continue
        if(index1.eq.0) found=.false.

        xx1=xto_nod-dsmall
        xx2=xto_nod+dsmall
        index2=0
        do 2015 ix=1,nx
          if(xi(ix).ge.xx1.and.xi(ix).le.xx2) then
            index2=ix
            goto 2016
          end if
 2015   continue
 2016   continue
        if(index2.eq.0) found=.false.

        yy1=yfrom_nod-dsmall
        yy2=yfrom_nod+dsmall
        indey1=0
        do 2020 iy=1,ny
          if(yi(iy).ge.yy1.and.yi(iy).le.yy2) then
            indey1=iy
            goto 2021
          end if
 2020   continue
 2021   continue
        if(indey1.eq.0) found=.false.

        yy1=yto_nod-dsmall
        yy2=yto_nod+dsmall
        indey2=0
        do 2025 iy=1,ny
          if(yi(iy).ge.yy1.and.yi(iy).le.yy2) then
            indey2=iy
            goto 2026
          end if
 2025   continue
 2026   continue
        if(indey2.eq.0) found=.false.

        if(.not.found) then
         write(66,3000) iflag,xx1,xx2,index1,index2,
     1                        yy1,yy2,indey1,indey2,
     2                        zz1,zz2,indez1,indez2
         write(*,3000)  iflag,xx1,xx2,index1,index2,
     1                        yy1,yy2,indey1,indey2,
     2                        zz1,zz2,indez1,indez2
         stop
        end if
 3000   format(/,' **** ERROR ****',/,' flag=',i3,
     &  '   Nodes not found',/,
     &  ' xmin    =',d12.4,' xmax    =',d12.4,/,
     &  ' x-index1=',i12  ,' x-index2=',i12,/,
     &  ' ymin    =',d12.4,' ymax    =',d12.4,/,
     &  ' y-index1=',i12  ,' y-index2=',i12,/,
     &  ' zmin    =',d12.4,' zmax    =',d12.4,/,
     &  ' z-index1=',i12  ,' z-index2=',i12)

        idum=indez1
        if(indez1.gt.indez2) then
          indez1=indez2
          indez2=idum
        end if
        idum=index1
        if(index1.gt.index2) then
          index1=index2
          index2=idum
        end if
        idum=indey1
        if(indey1.gt.indey2) then
          indey1=indey2
          indey2=idum
        end if
c
c  ...Assign nodes
c
        ncount=0
        do 3050 iz=indez1,indez2
          do 3040 iy=indey1,indey2
            do 3030 ix=index1,index2
              ncount=ncount+1
              nident(ncount+ilast)=(iz-1)*nndsl+(iy-1)*nx+ix
 3030       continue
 3040     continue
 3050   continue
        iend=ilast+ncount

      elseif(nln.eq.6) then
c
c  ...Irregular grid (triangular prisms)
c
        xx1=xfrom_nod-dsmall
        xx2=xto_nod+dsmall
        yy1=yfrom_nod-dsmall
        yy2=yto_nod+dsmall
        ncount=0
        do 4050 iz=indez1,indez2
          node1=(iz-1)*nndsl + 1
          node2=iz*nndsl
          do 4040 nnode=node1,node2
            if(x(nnode).lt.xx1) goto 4040 
            if(x(nnode).gt.xx2) goto 4040
            if(y(nnode).lt.yy1) goto 4040 
            if(y(nnode).gt.yy2) goto 4040
            ncount=ncount+1
            nident(ncount+ilast)=nnode
 4040     continue
 4050   continue
        iend=ilast+ncount

      end if
        
      return
      end

c***************************************************************************

      subroutine solve_sat(ntloop)

c***************************************************************************
c
c  ...Solve the matrix equation for a saturated domain
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
c
c  ...Assemble the global matrix and right side vector
c
      call set_matrix_sat(ntloop)
c
c  ...Perform decomposition
c
      call iluc
cws ...Set head values of inactive nodes at -9999.0 by default
      default_h=-9999.0d0
	do 451 i=1,nn
cws	   write(66,*) inactnode(i)
	   if(inactnode(i)) then
            k=iadpiv(i)
            r(k)=1.d30
            gb(i)=default_h*1.d30
            cu(i)=default_h
         endif
  451 continue
c
c  ...solve the matrix equation 'r c = g'
c    
c      maxit=min0(nn,2000)
      maxit=nn*10
      call orthomin(maxit,resx,resr)

      write(66,267) maxit
 267  format (' ORTHOMIN iterations done: ',i5)
 
      return
      end

c***************************************************************************

      subroutine set_matrix_sat(ntloop)

c***************************************************************************
c
c  ...Assemble the global matrix system for a saturated domain
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
c
c  ...initialize total flux vector and global coefficient matrix
c    
      call dzero(nn,gb(1),1)
      call dzero(ia(nn+1),r(1),1)

      do 2031 i=1,nntot_well
        do 2028 j=1,maxnb
          rowell(i,j)=0.0d0
 2028   continue
        gbwell(i)=0.0d0
 2031 continue
      do 2041 i=1,nntot_obsw
        do 2038 j=1,maxnb
          rowobs(i,j)=0.0d0
 2038   continue
        gbobs(i)=0.0d0
 2041 continue
c
c  ...Global assembly (porous media)
c
      mbal_flag=.false.

      call global_assembly_sat(ntloop)
c
c  ...Save rows of global matrix for back calculating nodal fluxes
c
      if(nwell.gt.0) then
        icount=0
        do 346 i=1,nwell
          do 343 j=1,nn_well(i)
            icount=icount+1
            n1=jq(i,j)
            istart=ia(n1)
            iend=ia(n1+1) - 1
            ik=0
            do 341 k=istart,iend
              ik=ik+1
              rowell(icount,ik)=r(k)
  341       continue
            gbwell(icount)=gb(n1)
  343     continue
  346   continue
      end if     

      if(nobsw.gt.0) then
        icount=0
        do 1346 i=1,nobsw
          do 1343 j=1,nn_obsw(i)
            icount=icount+1
            n1=jobs(i,j)
            istart=ia(n1)
            iend=ia(n1+1) - 1
            ik=0
            do 1341 k=istart,iend
              ik=ik+1
              rowobs(icount,ik)=r(k)
 1341       continue
            gbobs(icount)=gb(n1)
 1343     continue
 1346   continue
      end if     
c
c  ...2nd-type boundary contribution to flow matrix
c     and right-hand side vector
c    
      if (nbc2.gt.0) then
        do 1500 i=1,nbc2
          node=jbc2(i)
          gb(node)=gb(node)+fluxbc2(i)
 1500   continue
      end if
c
c  ...Contribution to flux vector from source/sink nodes
c    
c     If first time through then divide the well discharge
c     by the number of well nodes to get approximation
c     of discharge at each well node
c
      if(nwell.gt.0) then
        call soursink(ntloop)
      end if
c
c  ...Take care of the dirichlet nodes
c     use large values to swamp out other effects
c    
      if(ndc.gt.0) then
        do 316 i=1,ndc
          node=jm(i)
          k=iadpiv(node)
          r(k)=1.d30
          gb(node)=hbc1(i)*1.d30
          cu(node)=hbc1(i)
  316   continue
      endif

     

c     
c     ... Check wether inactive nodes exist
c     ... eliminate missing self-connection
c     
      do i = 1,nn
        k=iadpiv(i)
        if (r(k).EQ.0) r(k) = 1.d30
      end do


      return
      end

c***************************************************************************

      subroutine global_assembly_sat(ntloop)

c***************************************************************************
c
c  ...Compute the elemental matrices and assemble the
c     global matrix for the porous medium
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

     

c
c  ...Loop over hexahedral (brick) or prism elements
c   
      do 200 l=1,ne
c          if (dkxx(l).le.0) then
c              iprop(l)=0
c          endif
csd        if (iprop(l).gt.0) then
        izone=iprop(l)
c
c  ...Element lengths in x-y-z directions
c    
        if(nln.eq.8) then
          call elemdim(l,dx,dy,dz)
        elseif(nln.eq.6) then
          call coefpr(l,delt)
          el14=dlac(in(1,l),in(4,l))
          el25=dlac(in(2,l),in(5,l))
          el36=dlac(in(3,l),in(6,l))
          dz=(el14+el25+el36)/3.0d0
        end if
c
c  ...Precompute some quantities used in assembly
c    
        if(nln.eq.8) then
           elv=dx*dy*dz
           if (izone.gt.0) then
              aa1=dkxx(izone)*dy*dz/(2.d0*dx)
              aa2=dkyy(izone)*dx*dz/(2.d0*dy)
              aa3=dkzz(izone)*dx*dy/(2.d0*dz)
              aa10=stor(izone)*elv/(8.0d0*delta)
           else
              aa1=0.
              aa2=0.
              aa3=0.
              aa10=0.
           endif
        elseif(nln.eq.6) then
           elv=dz*delt
           if (izone.gt.0) then
              aa1=dkxx(izone)*elv*half
              aa2=dkyy(izone)*elv*half
              aa3=dkzz(izone)*2.0d0*delt/(3.d0*dz)
              aa10=stor(izone)*elv/(6.0d0*delta)
           else
              aa1=0.
              aa2=0.
              aa3=0.
              aa10=0.
           endif
        end if
c
c  ...Global flow matrix assembly
c
        if(steady_state) aa10=0.0d0
        if(.not.mbal_flag) then
          do 211 i=1,nln
            n1=in(i,l)
            do 212 j=1,nlnj
             n2=in(jloop(i,j),l)
             jj=jloop(i,j)
             call find(n1,n2,iband)
             if(iband.ne.0) then
               term=tw*(aa1*edxx(i,jj)+aa2*edyy(i,jj)+aa3*edzz(i,jj))    
               termc = aa10*eb(i,jj)
               r(iband) = r(iband)+term+termc
               term2= term*twratio
               gb(n1)= gb(n1) + (term2 + termc)*ci(n2)
             end if
  212       continue
  211     continue
        elseif(mbal_flag) then
          do 611 i=1,nln
            n1=in(i,l)
            if(ic(n1)) then
              sumleft=0.0d0
              sumright=0.0d0
              do 612 j=1,nlnj
                n2=in(jloop(i,j),l)
                jj=jloop(i,j)
                call find(n1,n2,iband)
                if(iband.ne.0) then
                  term=tw*(aa1*edxx(i,jj)+aa2*edyy(i,jj)+
     &                 aa3*edzz(i,jj))
                  termc = aa10*eb(i,jj)
                  sumleft = sumleft+(term+termc)*cu(n2)
                  term2= term*twratio
                  sumright = sumright + (term2 + termc)*ci(n2)
                end if
  612         continue
              vrv(n1)=vrv(n1)+(sumleft-sumright)
            end if
  611     continue
c
c  ...Total mass accumulation for time step
c
          if(.not.steady_state) then
            do 241 i=1,nln
              n1=in(i,l)
              do 242 j=1,nlnj
                n2=in(jloop(i,j),l)
                jj=jloop(i,j)
                call find(n1,n2,iband)
                if(iband.ne.0) then
                  termc = aa10*eb(i,jj)
                  totmacc_pm=totmacc_pm+termc*(cu(n2)-ci(n2))
                end if
  242         continue
  241       continue
          end if
c
c  ...If first time step, compute initial mass in system
c
          if(.not.steady_state.and.ntloop.eq.1) then
            do 251 i=1,nln
              n1=in(i,l)
              do 252 j=1,nlnj
                n2=in(jloop(i,j),l)
                jj=jloop(i,j)
                call find(n1,n2,iband)
                if(iband.ne.0) then
                  termc = aa10*eb(i,jj)
                  fmass_init=fmass_init+termc*ci(n2)
                end if
  252         continue
  251       continue
          end if

        end if
c
c  ...End loop over hexahedral (brick) elements
c    
c        endif
  200   continue
        return
        end
 
c***************************************************************************

      subroutine soursink(ntloop)

c***************************************************************************
c
c  ...Account for contribution to flux vector from source/sink nodes
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
      logical atbottom,attop
      dimension trans(maxwelln)
c    
c     If first time through then divide the well discharge
c     by the number of well nodes to get approximation
c     of discharge at each well node
c
c  ...For first time step:
c     assign flux to well node according to its volume of influence.
c     it is assumed that the volume of influence is proportional to the
c     vertical length of influence of well node.
c     this length is obtained by computing the vertical distance between
c     2 adjacent well nodes.
c
      dsmall=1.0d-5

      icount=0
      nxny=nx*ny
      if(.not.iter_flux.or.ntloop.eq.1) then
        write(66,9046) 
 9046   format(41('*'),/,2x,'Flux distribution for first iteration',/,
     &   41('*'),/,2x,'well',2x,'node',7x,'flux',/)
        do 8905 i=1,nwell
          hw(i)=0.0d0
          do 8903 j=1,nn_well(i)
            hw1(i)=hw1(i)+cu(jq(i,j))
 8903     continue
          hw1(i)=hw1(i)/dble(nn_well(i))
          n1=jq(i,1)
          n2=jq(i,nn_well(i))
          if(nn_well(i).gt.1) then
c
c  ...If well is not fully penetrating the domain, check
c     if its extremities are at the bottom or at the top
c     of the domain
c
              atbottom=.true.
              attop=.true.
              if(nn_well(i).ne.nz) then
                if(use_coord) then
                  if(z(n1).gt.zi(1)+dsmall) atbottom=.false.
                  if(z(n2).lt.zi(nz)-dsmall) attop=.false.
                 elseif(.not.use_coord) then
	            if(jslice(i,1).ne.1) atbottom=.false.
				if(jslice(i,nn_well(i)).ne.nz) attop=.false.
cws	            write(6,*) atbottom, attop
			  endif
	        endif
c
c  ...Compute transmissivity weighting
c
              total_trans=0.0d0
              do 8912 j=1,nn_well(i)
                trans(j)=0.0d0
                node=jq(i,j)
                if(j.eq.1) then
                  vlen=dabs( z(node+nxny) - z(node) ) * half
cws	            write(6,*) z(node+nxny), z(node)
cws	            write(6,*) dkeq_above(i,j)
                  trans(j)=trans(j) + vlen*dkeq_above(i,j)
                  if(.not.atbottom) then
                    vlen=  dabs( z(node-nxny) - z(node) ) / 2.d0
                    trans(j)=trans(j) + vlen*dkeq_below(i,j)
                  end if
                elseif(j.lt.nn_well(i)) then
                  vlen=dabs( z(node+nxny) - z(node) ) / 2.d0
                  trans(j)=trans(j) + vlen*dkeq_above(i,j)
                  vlen=  dabs( z(node-nxny) - z(node) ) / 2.d0
                  trans(j)=trans(j) + vlen*dkeq_below(i,j)
                elseif(j.eq.nn_well(i)) then
                  vlen=dabs( z(node-nxny) - z(node) ) / 2.d0
cws	            write(6,*) z(node-nxny), z(node)
                  trans(j)=trans(j) + vlen*dkeq_below(i,j)
                  if(.not.attop) then
                    vlen=dabs( z(node+nxny) - z(node) ) / 2.d0
                    trans(j)=trans(j) + vlen*dkeq_above(i,j)
                  end if
                end if
                total_trans=total_trans+trans(j)
 8912         continue
              qtot(i)=0.0d0
              do 8935 j=1,nn_well(i)
                icount=icount+1
                node=jq(i,j)
cws                write(6,*) flowrate(i), trans(j), total_trans
                pumpnod=flowrate(i)*trans(j)/total_trans
                gb(node)=gb(node)+pumpnod
                gbwellmb(icount)=pumpnod
                qtot(i)=qtot(i)+pumpnod
                write(66,9047) i,node,pumpnod
 9047           format(2i6,2x,d15.9)
 8935         continue
           elseif(nn_well(i).eq.1) then
            icount=icount+1
            gbwellmb(icount)=flowrate(i)
            gb(n1)=gb(n1)+flowrate(i)
            qtot(i)=flowrate(i)
cws            write(66,9047) i,n1,flowrate(i)
          end if
 8905   continue 
      elseif(ntloop.gt.1.and.iter_flux) then
        write(66,3489)
 3489   format(/,5x,'Extrapolated head at well(s)',/,33('-'),
     &    /,5x,'well',12x,'head')
c
c  ...Extrapolate heads at well(s).
c
        if(ntloop.eq.2) then
          tcons=dabs( dlog10(t(ntloop-1))/dlog(t(ntloop)) )
        elseif(ntloop.gt.2) then
          tcons=dlog10(t(ntloop)/t(ntloop-2))/
     &      dlog10(t(ntloop-1)/t(ntloop-2)) 
        end if
        do 7935 i=1,nwell
          if(ntloop.eq.2) then
            guess = tcons * (hw(i)-hw2(i)) + hw(i)
          else
            guess = tcons * (hw(i)-hw2(i)) + hw2(i)
          end if
          hw(i)=guess
          write(66,3490) i,hw(i)
 3490     format(i9,6x,d16.9)
          do 8934 j=1,nn_well(i)
            node=jq(i,j)
            k=iadpiv(node)
            r(k)=1.0d30
            gb(node)=1.0d30*hw(i)
            cu(node)=hw(i)
 8934     continue
 7935   continue
      end if

      return 
      end

c***************************************************************************

      subroutine wellits(ntloop)

c***************************************************************************
c
c  ...Loop for iterating well discharge and drawdowm
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'
c
c  ...One solution already performed, average out the hydraulic
c     head to impose a constant head at the well
c
      do 5555 i=1,nwell
        sum=0.0d0
        do 5545 j=1,nn_well(i)
          write(66,9034) jq(i,j),cu(jq(i,j))
 9034     format(i6,2x,d15.8)
          sum=sum+cu(jq(i,j))
 5545   continue
        hw(i)=sum/dble(nn_well(i))
        do 5550 j=1,nn_well(i)
          node=jq(i,j)
          k=iadpiv(node)
          r(k)=1.0d30
          gb(node)=1.0d30*hw(i)
          cu(node)=hw(i)
 5550   continue
 5555   continue
        numiter=50
c
c  ...Loop to solve if iteration (Jump out if no iteration wanted)
c
      do 5850 iter=1,numiter
        write(66,269)
  269   format (/31('*'),/,'Solution of the matrix equation'
     &    ,/,31('*'),/)
        write(66,5686) t(ntloop),iter
 5686   format(/46('*'),/2x,'Time:',d12.5,4x,'well head iteration:'
     &       ,i3,/,46('*'),//)
        maxit=nn
c        maxit=min0(nn,2000)
        if(iter.eq.1) call iluc
        call orthomin(maxit,resx,resr)
csd        write(66,266) maxit
  266   format('Number of ORTHOMIN iterations:',i4,/)
c
c  ...Compute total flux at wells
c
        icount=0
        do 5515 i=1,nwell
          qtot(i)=0.0d0
 5515   continue
        do 5577 i=1,nwell
          write(66,5512) i,hw(i)
 5512     format('Flux distribution for well',i3,/,29('-'),
     &    //,'Imposed head: ',d16.9,
     &    //,5x,'Node',10x,'flux')
          do 5573 j=1,nn_well(i)
            n1=jq(i,j)
            sum=0.0d0
            icount=icount+1
            istart=ia(n1)
            iend=ia(n1+1) - 1
            ik=0
            do 5569 k=istart,iend
              node=ja(k)
              ik=ik+1
              sum=sum + rowell(icount,ik)*cu(node)
 5569       continue
            flux=sum-gbwell(icount)
            gbwellmb(icount)=flux
            qtot(i) = qtot(i) + flux
            write(66,5516) n1,gbwellmb(icount)
 5516       format(4x,i5,3x,d16.9)
 5573     continue
          write(66,5534) i,qtot(i)
 5534     format(/,'Well',i3,
     &        ' total flux (calculated): ',d16.9,/)
 5577   continue
c
        iflag=0
        do 5870 i=1,nwell
          check=dabs(flowrate(i)-qtot(i))
          if(check.gt.epsq(i)) iflag=1
 5870   continue
        if(iflag.eq.0) goto 9200
c
c  ...Correct the value of hw(i)
c 
        do 5950 i=1,nwell
          if(iter.eq.1.and.ntloop.eq.1) then
            hw1(i)=ci(jq(i,1))
            qtot2(i)=(1.0d0-hw1(i))/(1.0d0-hw(i))*flowrate(i)
          end if
          hwnext=(hw(i)-hw1(i))/(qtot(i)-qtot2(i)) *
     &      (flowrate(i)-qtot2(i)) + hw1(i)
          hw1(i)=hw(i)
          hw(i)=hwnext
          qtot2(i)=qtot(i)
 5950   continue
c
        do 5947 i=1,nwell
          do 5945 j=1,nn_well(i)
            node=jq(i,j)
            k=iadpiv(node)
            r(k)=1.0d30
            gb(node)=1.0d30*hw(i)
            cu(node)=hw(i)
 5945     continue
 5947   continue

 5850 continue

 9200 continue

      return
      end

c***************************************************************************
 
      subroutine masbal(ntloop,time)

c***************************************************************************
c
c  ...Compute the fluid mass balance
c
c***************************************************************************

      include 'tbc.prm'
      include 'tbc.dim'

      dsmall = 1.0d-5
c
c  ...Initialise vector vrv containing nodal flux at dirichlet nodes
c     and vector rq containing flux at 2nd-type nodes
c
      do 10 i=1,nn
        vrv(i)=0.0d0
        rq(i)=0.0d0
  10  continue
      totmacc_pm=0.0d0
      fmass_init=0.0d0
      if(ntloop.eq.1) then
        cumul_mass=0.0d0
        cumul_err=0.0d0
      end if
c
c  ...Compute flux at first-type nodes + mass accumulation
c
      mbal_flag=.true.

      call global_assembly_sat(ntloop)

c
c  ...Sum up flux at first-type nodes
c
      qabs=0.0d0
      qfirstpos=0.0d0
      qfirstneg=0.0d0
      do 3050 i=1,nn
        if(vrv(i).gt.0.0d0) qfirstpos=qfirstpos+vrv(i)
        if(vrv(i).lt.0.0d0) qfirstneg=qfirstneg+vrv(i)
        qabs=qabs+dabs(vrv(i))
 3050 continue
      write(42) ndc
      if(ndc.gt.0) then
        write(42) (jm(i),vrv(jm(i)),i=1,ndc)
      end if
      if(transport) then
        do 4500 i=1,ndc
          fluxdch(i)=vrv(jm(i))
 4500   continue
      end if
c
c  ...Sum up flux at second-type nodes
c
      qsecpos=0.0d0
      qsecneg=0.0d0
      if(nbc2.gt.0) then
        do 4550 i=1,nbc2
          if(fluxbc2(i).gt.0.0d0) qsecpos=qsecpos+fluxbc2(i)
          if(fluxbc2(i).lt.0.0d0) qsecneg=qsecneg+fluxbc2(i)
 4550   continue
      end if

      write(42) nbc2
      if(nbc2.gt.0) then
        do 3065 i=1,nbc2
          write(42) jbc2(i),fluxbc2(i)
 3065   continue
c
c  ...Store all fluxes in vector vrv().
c
        if(outfc) then
          do 3068 i=1,nbc2
            vrv(jbc2(i)) = vrv(jbc2(i)) + fluxbc2(i)
 3068     continue
        end if
      end if
c
c  ...Output fluid flux at specified nodes
c
      if(outfc) then
        write(62,*) ' Fluid flux, time:', time
        do 6360 i=1,noutfc
          write(62,6820) ioutfc(i),vrv(ioutfc(i))
 6360   continue
      end if
 6820 format(i7,2x,e12.5)

c
c  ...Flux at wells
c
      qwellpos=0.0d0
      qwellneg=0.0d0
      if(nwell.gt.0) then
        do 4050 i=1,nwell
          if(qtot(i).gt.0.0d0) qwellpos=qwellpos+qtot(i)
          if(qtot(i).lt.0.0d0) qwellneg=qwellneg+qtot(i)
          qabs=qabs+dabs(qtot(i))
 4050   continue
      end if

      qnet=qfirstpos+qfirstneg+qsecpos+qsecneg+qwellpos+qwellneg
c
c  ...Check if mass balance is outputted
c
      if(kpmasb.gt.0) then
        itemp=ntloop/kpmasb
        temp2=dble(itemp)
        temp=dble(ntloop)/dble(kpmasb)
        check=temp-temp2
      else
        check=dsmall*1.0d5
        if(kpmasb.lt.0) check=0.1d0*dsmall
      end if
      if(target_reached .or. (check.lt.dsmall)) then
        if(kpmasb.ne.0) then
          if(steady_state) then
            write(66,6058)
          else
            write(66,6059) time
          end if
          write(66,6060) qfirstpos,qfirstneg,qsecpos,
     &              qsecneg,qwellpos,qwellneg
          write(66,6075) qnet
        end if
      end if
 6058 format(/,12('*'),/,'Mass balance'/,12('*'))
 6059 format(/,35('*'),/,'Mass balance, Time: ',d15.9,/,35('*'))
 6060 format('Flow at boundaries and wells [L**3/T]',/,37('-'),
     & /,'Inflow at first-type nodes:',t36,d15.9,
     & /,'Outflow at first-type nodes:',t36,d15.9,
     & /,'Inflow at second-type nodes:',t36,d15.9,
     & /,'Outflow at second-type nodes:',t36,d15.9,
     & /,'Inflow at injection wells:',t36,d15.9,
     & /,'Inflow at pumping wells:',t36,d15.9)
 6075 format('Net flow (inflow-outflow):',t36,d15.9)

      if(.not.steady_state) then
        accumulation=totmacc_pm
        qdiff=totmacc_pm-qnet
        qnorm=dabs(qdiff)/(qabs+dabs(totmacc_pm))
        cumul_mass=cumul_mass+totmacc_pm
        cumul_err=cumul_err+qdiff

        ratio_tstep=dabs(qdiff/totmacc_pm)*100.d0
        ratio_cumul=dabs(cumul_err/cumul_mass)*100.d0
        if(target_reached .or. (check.lt.dsmall)) then
          if(kpmasb.ne.0) then
            write(66,6061) totmacc_pm,
     &          accumulation,qdiff,qnorm
            write(66,6062) ratio_tstep,ratio_cumul
          end if
        end if

 6061   format(/,'Rate of mass accumulation for time step [L**3/T]',
     &  /,48('-'),/,'Accumulation rate in porous media:',t36,d15.9,
     &  /,'Net accumulation in domain:',t36,d15.9,
     &  //,'Time step mass balance error',/,'(Accumulation - net',
     &  ' flow):',t36,d15.9,/,'Time step normalized error:',t36,d15.9)
 6062   format(/,'Transient simulation - mass balance [L**3/T]',/,
     &  42('-'),/,'Time step relative error (error/mass accumulated)',
     &   t60,f15.9,' %',/,'Cumulative relative error ',
     &   '(error/mass accumulated)',t60,f15.9,' %')
      else
        if(target_reached .or. (check.lt.dsmall)) then
          if(kpmasb.ne.0) then
            write(66,6070)
          end if
        end if
 6070   format(/'Steady-state simulation',/,
     &  'No accumulation in domain')
      end if

      return
      end
 
c***************************************************************************
 
      subroutine compute_flux(kprint_flag)

c***************************************************************************
c
c  ...Compute the elemental darcy fluxes for 3D elements
c
c***************************************************************************
 
      include 'tbc.prm'
      include 'tbc.dim'
      

      if(nln.eq.8) then
          do 7320 i=1,ne
            izone=iprop(i)
            h1=cu(in(1,i))
            h2=cu(in(2,i))
            h3=cu(in(3,i))
            h4=cu(in(4,i))
            h5=cu(in(5,i))
            h6=cu(in(6,i))
            h7=cu(in(7,i))
            h8=cu(in(8,i))
            dx=x(in(2,i))-x(in(1,i))
            dy=y(in(4,i))-y(in(1,i))
            dz=z(in(5,i))-z(in(1,i))
cws	write(66,*) izone, dkxx(izone)
cws            if (iprop(i).ne.0) then
            vx(i)=dkxx(izone)*(h1+h4+h5+h8-h2-h3-h6-h7)/(4.0d0*dx)
            vy(i)=dkyy(izone)*(h1+h2+h5+h6-h3-h4-h7-h8)/(4.0d0*dy)
            vz(i)=dkzz(izone)*(h1+h2+h3+h4-h5-h6-h7-h8)/(4.0d0*dz)
cws            h1=cu(in(1,i))
cws            h2=cu(in(2,i))
cws            h3=cu(in(3,i))
cws            h4=cu(in(4,i))
cws            h5=cu(in(5,i))
cws            h6=cu(in(6,i))
cws            h7=cu(in(7,i))
cws            h8=cu(in(8,i))
cws            dx=x(in(2,i))-x(in(1,i))
cws            dy=y(in(4,i))-y(in(1,i))
cws            dz=z(in(5,i))-z(in(1,i))
cws            elseif(iprop(i).eq.0) then      ! impermeable element
cws              vx(i)=0.0
cws              vy(i)=0.0
cws              vz(i)=0.0      
cws           end if
               
 7320     continue
      elseif(nln.eq.6) then
        do 7330 i=1,ne
          izone=iprop(i)
          x2=x(in(2,i))-x(in(1,i))
          x3=x(in(3,i))-x(in(1,i))
          y2=y(in(2,i))-y(in(1,i))
          y3=y(in(3,i))-y(in(1,i))
          delt2=x2*y3-x3*y2
          delt2inv=1.0d0/delt2
          beta1=(y2-y3)*delt2inv
          beta2=y3*delt2inv
          beta3=-y2*delt2inv
          gamma1=(x3-x2)*delt2inv
          gamma2=-x3*delt2inv
          gamma3=x2*delt2inv
          deltaz=z(in(4,i))-z(in(1,i))
          h1=cu(in(1,i))
          h2=cu(in(2,i))
          h3=cu(in(3,i))
          h4=cu(in(4,i))
          h5=cu(in(5,i))
          h6=cu(in(6,i))
          betah=beta1*h1+beta2*h2+beta3*h3+
     &          beta1*h4+beta2*h5+beta3*h6
          gammah=gamma1*h1+gamma2*h2+gamma3*h3+
     &           gamma1*h4+gamma2*h5+gamma3*h6
             if (iprop(i).ne.0) then
          vx(i)=-1.0d0*dkxx(izone)*betah*half
          vy(i)=-1.0d0*dkyy(izone)*gammah*half
          vz(i)=-1.0d0*dkzz(izone)*(h4+h5+h6-h1-h2-h3)/(3.0d0*deltaz)
          elseif(iprop(i).eq.0) then      ! impermeable element
              vx(i)=0.0
              vy(i)=0.0
              vz(i)=0.0      
           end if
 7330   continue
      end if

      if(kprint_flag.eq.1) then
        write(43) (vx(i),vy(i),vz(i),i=1,ne)
        if(echo_to_output) then
          write(66,8300)
          write(66,8301) (i,vx(i),vy(i),vz(i),i=1,ne)
        end if
      end if
 8300 format(/,'Computed darcy fluxes',/,21('-'),
     &   //,5x,'Element - vx - vy - vz'/)
 8301 format(2(i6,3(1x,d12.5)))

      return
      end



c***************************************************************************
      subroutine calcnodev
c***************************************************************************
c
      include 'tbc.prm'
      include 'tbc.dim'
 
      common /massbal/ talt,production,dnodev,node
      double precision production(maxreac,maxsp),dnodev(maxnn)

c  ... Volume of cells
c
      do 5678 i=1,nn
        dnodev(i)=0.d0
5678  continue

      if(nln.eq.8) then
        do 7423 l=1,ne
          call elemdim(l,dx,dy,dz)
          v=dx*dy*dz
          do 3648 i=1,nln
            node=in(i,l)
            if(icc(node).ne.1) dnodev(node)=dnodev(node)+v/8.d0
3648      continue
7423    continue
      else
        stop
      endif
c      do 2874 i=1,nn
c        write(65,*) i,dnodev(i)
c2874  continue
      return
      end


